/*
MiniTLS - A super trimmed down TLS/SSL Library for embedded devices
Author: Donatien Garnier
Copyright (C) 2013-2014 AppNearMe Ltd

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*//**
 * \file buffer_network.c
 * \copyright Copyright (c) AppNearMe Ltd 2013
 * \author Donatien Garnier
 */

#include "core/fwk.h"

#include "buffer_network.h"

#define VOID
#define ENSURE_READ_LENGTH(b, n, r) do{ if(b->end - b->start < n) { return r; } }while(0);
#define ENSURE_WRITE_LENGTH(b, n) do{ if(b->bufdata + b->size - b->end < n) { return; } }while(0);

uint8_t buffer_nu8_read(buffer_t* buffer)
{
  uint8_t hu8;
  ENSURE_READ_LENGTH(buffer, 1, 0);
  hu8 = buffer->start[0];
  buffer->start++;
  return hu8;
}

uint16_t buffer_nu16_read(buffer_t* buffer)
{
  uint16_t hu16;
  ENSURE_READ_LENGTH(buffer, 2, 0);
  hu16 = (buffer->start[0] << 8) | buffer->start[1];
  buffer->start+=2;
  return hu16;
}

uint32_t buffer_nu24_read(buffer_t* buffer)
{
  uint32_t hu24;
  ENSURE_READ_LENGTH(buffer, 3, 0);
  hu24 = (buffer->start[0] << 16) | (buffer->start[1] << 8) | buffer->start[2];
  buffer->start+=3;
  return hu24;
}

uint32_t buffer_nu32_read(buffer_t* buffer)
{
  uint32_t hu32;
  ENSURE_READ_LENGTH(buffer, 4, 0);
  hu32 = (buffer->start[0] << 24) | (buffer->start[1] << 16) | (buffer->start[2] << 8) | buffer->start[3];
  buffer->start+=4;
  return hu32;
}

uint64_t buffer_nu64_read(buffer_t* buffer)
{
  uint64_t hu64;
  ENSURE_READ_LENGTH(buffer, 8, 0);
  hu64 = ((uint64_t)buffer->start[0] << 56) | ((uint64_t)buffer->start[1] << 48) | ((uint64_t)buffer->start[2] << 40) | ((uint64_t)buffer->start[3] << 32)
      | (buffer->start[4] << 24) | (buffer->start[5] << 16) | (buffer->start[6] << 8) | buffer->start[7];
  buffer->start+=8;
  return hu64;
}

void buffer_nbytes_read(buffer_t* buffer, uint8_t* data, size_t size)
{
  ENSURE_READ_LENGTH(buffer, size, VOID);
  memcpy(data, buffer->start, size);
  buffer->start+=size;
}

void buffer_n_discard(buffer_t* buffer, size_t size)
{
  ENSURE_READ_LENGTH(buffer, size, VOID);
  buffer->start+=size;
}

uint8_t* buffer_current_read_position(buffer_t* buffer)
{
  return buffer->start;
}

size_t buffer_get_read_offset(buffer_t* buffer)
{
  return buffer->start - buffer->bufdata;
}

void buffer_set_read_offset(buffer_t* buffer, size_t off)
{
  if( buffer->bufdata + off > buffer->end )
  {
    return;
  }
  buffer->start = buffer->bufdata + off;
}

void buffer_nu8_write(buffer_t* buffer, uint8_t hu8)
{
  ENSURE_WRITE_LENGTH(buffer, 1);
  buffer->end[0] = hu8;
  buffer->end++;
}

void buffer_nu16_write(buffer_t* buffer, uint16_t hu16)
{
  ENSURE_WRITE_LENGTH(buffer, 2);
  buffer->end[0] = (hu16 >> 8) & 0xFF;
  buffer->end[1] = hu16 & 0xFF;
  buffer->end+=2;
}

void buffer_nu24_write(buffer_t* buffer, uint32_t hu24)
{
  ENSURE_WRITE_LENGTH(buffer, 3);
  buffer->end[0] = (hu24 >> 16) & 0xFF;
  buffer->end[1] = (hu24 >> 8) & 0xFF;
  buffer->end[2] = hu24 & 0xFF;
  buffer->end+=3;
}

void buffer_nu32_write(buffer_t* buffer, uint32_t hu32)
{
  ENSURE_WRITE_LENGTH(buffer, 4);
  buffer->end[0] = (hu32 >> 24) & 0xFF;
  buffer->end[1] = (hu32 >> 16) & 0xFF;
  buffer->end[2] = (hu32 >> 8) & 0xFF;
  buffer->end[3] = hu32 & 0xFF;
  buffer->end+=4;
}

void buffer_nu64_write(buffer_t* buffer, uint64_t hu64)
{
  ENSURE_WRITE_LENGTH(buffer, 8);
  buffer->end[0] = (hu64 >> 56) & 0xFF;
  buffer->end[1] = (hu64 >> 48) & 0xFF;
  buffer->end[2] = (hu64 >> 40) & 0xFF;
  buffer->end[3] = (hu64 >> 32) & 0xFF;
  buffer->end[4] = (hu64 >> 24) & 0xFF;
  buffer->end[5] = (hu64 >> 16) & 0xFF;
  buffer->end[6] = (hu64 >> 8) & 0xFF;
  buffer->end[7] = hu64 & 0xFF;
  buffer->end+=8;
}

void buffer_nbytes_write(buffer_t* buffer, const uint8_t* data, size_t size)
{
  ENSURE_WRITE_LENGTH(buffer, size);
  memcpy(buffer->end, data, size);
  buffer->end+=size;
}

void buffer_n_skip(buffer_t* buffer, size_t size)
{
  ENSURE_WRITE_LENGTH(buffer, size);
  buffer->end+=size;
}

uint8_t* buffer_current_write_position(buffer_t* buffer)
{
  return buffer->end;
}
