//---------------------------------------------------------------
//  液晶表示器 AQM1602XA-RN-GBW 用クラス，メンバ関数の定義
//
//  Nucleo の場合のデフォルトのピン
//      D14 ---- 液晶モジュールの SDA ピン
//      D15 ---- 液晶モジュールの SCL ピン
//
//  2020/04/16, Copyright (c) 2020 MIKAMI, Naoki
//---------------------------------------------------------------

#include "AQM1602.hpp"

namespace Mikami
{
    // コンストラクタ
    Aqm1602::Aqm1602(PinName sda, PinName scl, uint32_t clock,
                     bool cursor, bool blink)
        : i2cPtr_(new I2C(sda, scl)), i2c_(*i2cPtr_)
    {   Init(clock, cursor, blink); }

    Aqm1602::Aqm1602(I2C &i2c, uint32_t clock,
                     bool cursor, bool blink)
        : i2cPtr_(NULL), i2c_(i2c)
    {   Init(clock, cursor, blink); }

    // 表示のクリア
    void Aqm1602::Clear() const
    {
        WriteCmd(0x01);
        wait_ms(50);
    }

    // 文字列の書込み
    void Aqm1602::WriteString(const string str) const
    {
        int length = min(str.length(), (size_t)N_CHR);
        for (int n=0; n<length; n++) WriteChar(str[n]);
    }

    // 指定した位置から文字列の書込み
    void Aqm1602::WriteStringXY(const string str,
                                uint8_t x, uint8_t y) const
    {
        SetXY(x, y);
        WriteString(str);
    }

    // 指定した行のクリア
    void Aqm1602::ClearLine(uint8_t line) const
    {
        SetXY(0, line);
        for (int n=0; n<N_CHR; n++) WriteString(" ");
    }

    // コントラストの設定
    void Aqm1602::SetContrast(uint8_t c) const
    {
        WriteCmd(0x39);                     // 拡張コマンドへ
        WriteCmd(0x70 | (c & 0x0f));        // 下位 4 ビット
        WriteCmd(0x5C | ((c >> 4) & 0x03)); // 上位 2 ビット
        WriteCmd(0x38);                     // ノーマル･コマンドへ
    }

    // 初期化
    void Aqm1602::Init(uint32_t clock, bool cursor, bool blink)
    {
        if (clock != 100000) i2c_.frequency(clock);

        wait_ms(100);

        // 初期化のためのコマンド送信手順は，秋月電子の AE-AQM1602A(KIT) の
        // データシートの記載に従った
        connected_ = WriteCmd(0x38); // データ長 8-bit, 2-line, 5×8 dots
        if (!connected_)
        {
            fprintf(stderr, "\r\nLCD AQM1602 not connected\r\n");
            return;
        }

        WriteCmd(0x39);         // 拡張命令モードへ
        WriteCmd(0x14);         // 内部発信周波数
        WriteCmd(0x70 | 0x00);  // コントラスト設定
        WriteCmd(0x54 | 0x02);  // Power/ICON/Contrast control
        WriteCmd(0x6C);         // Follower control
        wait_ms(200);

        WriteCmd(0x38);         // 通常の命令モードへ復帰,
                                // データ長：8-bit, 2-line, 5×8 dots
//        WriteCmd(0x01);         // クリア，これを入れると正常に動作しない
        WriteCmd(0x0C | (cursor << 1) | blink);

        Clear();
    }

    // コマンドとデータの送信
    bool Aqm1602::LcdTx(uint8_t cmdData, uint8_t data) const
    {
        // このメンバ関数を割込み処理で使う場合，以下の処理ではうまく動作しない
/*
        int rt = i2c_.write(LCD_ADDRESS_, (char []){cmdData, data}, 2);
        wait_us(30);
        return (rt == 0) ? true : false;
*/
        // 以下の処理の場合は，このメンバ関数を割込み処理で使っても大丈夫
        i2c_.start();
        if (1 != i2c_.write(LCD_ADDRESS_)) return false;
        if (1 != i2c_.write(cmdData)) return false;
        if (1 != i2c_.write(data)) return false;
        i2c_.stop();
        wait_us(30);
        return true;
    }
}