//---------------------------------------------------------------
//  液晶表示器 AQM1602XA-RN-GBW 用クラス，ヘッダ
//
//  Nucleo の場合のデフォルトのピン
//      D14 ---- 液晶モジュールの SDA ピン
//      D15 ---- 液晶モジュールの SCL ピン
//
//  割込みサービス･ルーチン内に記述した場合でも動作する
//  このプログラムを作った際の mbed のリビジョン：Rev.172
//
//  2020/04/16, Copyright (c) 2020 MIKAMI, Naoki
//---------------------------------------------------------------

#include "mbed.h"
#include <string>

#ifndef AQM1602I2C_HPP
#define AQM1602I2C_HPP

namespace Mikami
{
    class Aqm1602
    {
    public:
        // コンストラクタ
        // SDA, SCA のデフォルト値は Nucleo に対応
        Aqm1602(PinName sda = D14,          // SDA
                PinName scl = D15,          // SCL
                uint32_t clock = 100000,    // クロック: 100 kHz
                bool cursor = false,        // カーソル:  off
                bool blink = false);        // 点滅:      off

        Aqm1602(I2C &i2c,                   // I2C オブジェクトの参照
                uint32_t clock = 100000,    // クロック: 100 kHz
                bool cursor = false,        // カーソル:  off
                bool blink = false);        // 点滅:      off

        virtual ~Aqm1602()
        {   if (NULL != i2cPtr_) delete i2cPtr_;    }

        // 液晶表示器が接続されていない場合 false を返す
        bool IsConnected() const { return connected_; }
        
        // 表示のクリア
        void Clear() const;
        
        // コマンドの送信
        bool WriteCmd(uint8_t cmd) const { return LcdTx(0x00, cmd); }
        
        // １文字の書込み
        bool WriteChar(char data) const { return LcdTx(0x40, data); }
        
        // 表示位置の設定, x: 0 ～ 15, y: 0, 1
        void SetXY(uint8_t x = 0, uint8_t y = 0) const
        { WriteCmd(x + y*0x40 | 0x80);}
        
        // 文字列の書込み
        void WriteString(const string str) const;
        
        // 指定した位置から文字列の書込み
        void WriteStringXY(const string str, uint8_t x, uint8_t y) const;

        // 数値の書込み
        template <typename T>
        void WriteValue(const char fmt[], T value) const
        {
            char str[N_CHR+1];
            sprintf(str, fmt, value);
            WriteString(str);
        }

        // 指定した位置から数値の書込み
        template <typename T>
        void WriteValueXY(const char fmt[], T value,
                          uint8_t x, uint8_t y) const
        {
            SetXY(x, y);
            WriteValue(fmt, value);
        }

        // 指定した行のクリア
        void ClearLine(uint8_t line) const;

        // コントラストの設定
        void SetContrast(uint8_t c) const;

    private:
        // AQM1602A のアドレス
        //      左詰め 7-bit アドレス
        static const uint8_t LCD_ADDRESS_ = 0x7C;
        static const uint8_t N_CHR = 16;

        I2C *const i2cPtr_; // I2C オブジェクトのポインタ
        I2C &i2c_;          // I2C オブジェクトの参照
        bool connected_;    // false: 液晶表示器は接続されていない

        // 初期化
        void Init(uint32_t clock, bool cursor, bool blink);
        
        // コマンドとデータの送信
        bool LcdTx(uint8_t cmdData, uint8_t data) const;

        // コピー･コンストラクタと代入演算子関数の禁止
        Aqm1602(const Aqm1602&);
        Aqm1602& operator=(const Aqm1602&);
    };
}
#endif  // AQM1602I2C_HPP