//-------------------------------------------------------
//  Class for LCD, AQM0802A (Header)
//
//  Default pin assignments
//      D14  SDA ---- to pin4 of LCD module
//      D15  SCL ---- to pin3 of LCD module
//
//  Assignment of I2C ports
//                SDA                SCL
//      I2C1   PB_7 or PB_9(D14)  PB_6(D10) or PB_8(D15)
//      I2C2   PB_3(D3)           PB_10(D6)
//      I2C3   PB_4(D5) or PC_9   PA_8(D7)
//
//  2014/12/21, Copyright (c) 2014 MIKAMI, Naoki
//-------------------------------------------------------

#ifndef AQM0802I2C_HPP
#define AQM0802I2C_HPP

#include "mbed.h"

namespace Mikami
{
    class Aqm0802A
    {
    public:
        // Constructor
        Aqm0802A(PinName sda = D14,        // SDA
                 PinName scl = D15,        // SCL
                 uint32_t clock = 100000,  // clock: 100 kHz
                 bool cursor = false,      // cursor:  off
                 bool blink = false);      // blink:   off

        // All clear
        void Clear();
        // Send command
        void WriteCmd(uint8_t cmd) { LcdTx(0x00, cmd); }
        // Write character (Not 0x80 but 0x40)
        void WriteChar(char data) { LcdTx(0x40, data); }
        // Specify display position, x: 0 - 7, y: 0, 1
        void SetXY(uint8_t x = 0, uint8_t y = 0)
        { WriteCmd(x + y*0x40 | 0x80);}
        // Write string
        void WriteString(const char str[]);
        // Write string from specified position
        void WriteStringXY(const char str[], uint8_t x, uint8_t);
        // Clear of specified line
        void ClearLine(uint8_t line)
        { WriteStringXY("        ", 0, line); }
        void SetContrast(uint8_t c);

    private:
        // Slave address of AQM0802A
        //      left-justified 7-bit address
        static const uint8_t LCD_ADDRESS_ = 0x7C;

        I2C i2c_;               // Object of I2C

        void LcdTx(uint8_t cmdData, uint8_t data);

        // Forbid to use copy constructor
        Aqm0802A(const Aqm0802A&);
        // Forbid to use substitution operator
        Aqm0802A& operator=(const Aqm0802A&);
    };
}
#endif  // AQM0802I2C_HPP

