//------------------------------------------------------------------
//  温度センサ ADT7410 用の ADT7410 クラスのヘッダ
//      温度の分解能のデフォルト値: 13 ビット
//
//  割込みサービス･ルーチン内に記述した場合に動作しない点を修正した
//  このプログラムを作った際の mbed のリビジョン：Rev.172
//
//  2020/04/16, Copyright (c) 2020 MIKAMI, Naoki
//------------------------------------------------------------------

#ifndef ADT7410_HPP
#define ADT7410_HPP

#include "mbed.h"

namespace Mikami
{
    class ADT7410
    {
    public:
        // デフォルト･コンストラクタ
#if defined(STM32F4) || defined(STM32L0) || defined(__STM32F3xx_H)
        ADT7410(PinName sda = D14,      // SDA
                PinName scl = D15,      // SCL
                uint8_t addr = 0x90);   // I2C bus address
// デフォルト･コンストラクタは Nucleo のみ
#else
        ADT7410(PinName sda,            // SDA
                PinName scl,            // SCL
                uint8_t addr = 0x90);   // I2C bus address
#endif
        // 外部の I2C オブジェクトを与えるコンストラクタ
        ADT7410(I2C &i2c,               // 外部の I2C オブジェクト
                uint8_t addr = 0x90);   // I2C bus address

        // デストラクタ
        virtual ~ADT7410()
        {   if (NULL != i2cPtr_) delete i2cPtr_;    }

        // コンフィギュレーション･レジスタの設定
        void SetConfig(char val) const;

        // コンフィギュレーション･レジスタの読み込み
        // ADT7410 のデータシートの Figure 17 参照
        uint8_t GetConfig() const { return GetReg(CONFIG_); }

        // 温度の読み込み
        virtual float Read() const;
        
        // Read() の代わりに使える演算子
        virtual operator float() const { return Read(); }

        // リセット
        void Reset() const;

    protected:
        const uint8_t ADDR_;    // ADT7410 の左詰めのアドレス

        // ADT7410 の内部のレジスタのアドレス
        enum Reg { TEMPER_ = 0x00,  // 温度レジスタの上位バイト
                   ST_     = 0x02,  // ステータス･レジスタ
                   CONFIG_ = 0x03,  // コンフィギュレーション･レジスタ
                   ID_     = 0x0B,  // ID, 0xCX
                   RESET_  = 0x2F}; // ソフトウェア･リセット

        // ADT7410 内部レジスタのアドレスの送信
        // "STOP" を送らずにふたたび "START" を送る，つまりリスタート･コンディションに
        // する場合は，"repeated" を "true" にする
        void TxRegAddr(uint8_t reg, bool repeated = false) const;
        
        // ステータス･レジスタの読み込み
        uint8_t GetStatus() const { return GetReg(ST_); }

        // ADT7410 の内部レジスタの 1 バイトの読み込み
        uint8_t GetReg(Reg addr) const;

    private:
        I2C *const i2cPtr_; // I2C オブジェクトのポインタ

    protected:
        I2C &i2c_;          // I2C オブジェクトの参照

    private:
        // コピー･コンストラクタと代入演算子関数の禁止
        ADT7410(const ADT7410&);
        ADT7410& operator=(const ADT7410&);
    };
}
#endif  // ADT7410_HPP