//--------------------------------------------------------------
//  ADT7410 クラスのメンバ関数の定義
//
//  2020/04/16, Copyright (c) 2020 MIKAMI, Naoki
//--------------------------------------------------------------

#include "ADT7410.hpp"

namespace Mikami
{
    // コンストラクタ
    ADT7410::ADT7410(PinName sda, PinName scl, uint8_t addr)
        : ADDR_(addr), i2cPtr_(new I2C(sda, scl)), i2c_(*i2cPtr_)
    {   Reset(); }

    // 外部の I2C オブジェクトを与えるコンストラクタ
    ADT7410::ADT7410(I2C &i2c, uint8_t addr)
        : ADDR_(addr), i2cPtr_(NULL), i2c_(i2c)
    {   Reset(); }

    // コンフィギュレーション･レジスタの設定
    // ADT7410 のデータシートの Figure 15 参照
    void ADT7410::SetConfig(char val) const
    {
        TxRegAddr(CONFIG_, true);
        i2c_.write(val);
        i2c_.stop();
    }

    // 温度の読み込み
    // ADT7410 のデータシートの Figure 18 参照
    float ADT7410::Read() const
    {
        TxRegAddr(TEMPER_, true);
        i2c_.start();
        i2c_.write(ADDR_ | 0x01);
        uint8_t valH = i2c_.read(1);
        int16_t value = (valH << 8) | i2c_.read(0);
        i2c_.stop();

        if ( (GetConfig() & 0x80) == 0x00 )
            value &= 0xfff8;        // 分解能: 13 ビットの場合
        return value/128.0f;
    }

    // リセット
    // ADT7410 のデータシート Rev.A の p.19 参照
    void ADT7410::Reset() const
    {
        TxRegAddr(RESET_);
        wait_ms(250);
        // 分解能: 13 ビット，continuous 変換モード に設定
        SetConfig(0x00);
    }
    
    // ADT7410 内部レジスタのアドレスの送信
    void ADT7410::TxRegAddr(uint8_t reg, bool repeated) const
    {
        i2c_.start();
        i2c_.write(ADDR_);
        i2c_.write(reg);
        if (!repeated) i2c_.stop(); 
    }

    // ADT7410 の内部レジスタの 1 バイトの読み込み
    uint8_t ADT7410::GetReg(Reg addr) const
    {
        TxRegAddr(addr, true);
        i2c_.start();
        i2c_.write(ADDR_ | 0x01);
        uint8_t val = i2c_.read(0);
        i2c_.stop();
        return val;
    }
}