//--------------------------------------------------------------
//  Class for ADT7310 on one-shot mode
//      Default: 13-bit resolution
//  2017/10/28, Copyright (c) 2017 MIKAMI, Naoki
//--------------------------------------------------------------

#include "ADT7310.hpp"

namespace Mikami
{
    // Constructor
    ADT7310::ADT7310(PinName ss,
                     PinName mosi, PinName miso, PinName sck)
        : ss_(ss, 1), spi_(mosi, miso, sck)
    {
        Reset();
        spi_.format(8, 3);  // mode: 8 bits, POL = 1, PHA = 1
        SetConfigReg(0x00); // Clear configuration register.
    }

    // Software reset
    void ADT7310::Reset()
    {
        Select();
        for (int n=0; n<4; n++) SendData(0xFF);
        Deselect();
        wait_us(500);    
    }

    // Set configuration register
    void ADT7310::SetConfigReg(uint8_t value)
    {
        Select();
        SendWrite(CONFIG);  // for write to config. reg.
        SendData(value);
        Deselect();
    }
    
    // Set resolution
    void ADT7310::SetResolution16(bool res)
    {
        uint8_t config = GetConfigReg() & 0x7f;
        if (res) SetConfigReg(config | 0x80);
        else     SetConfigReg(config);
    }

    // Get specified register
    uint8_t ADT7310::GetReg(Reg myReg)
    {
        Select();
        SendRead(myReg);
        uint8_t value = SendData(0x00);
        Deselect();
        return value;
    }

    // Read data and translate to temperature
    float ADT7310::Read()
    {
        while ((GetStatusReg() & 0x80) != 0x00) {}
        
        Select();
        SendRead(DATA);
        int16_t data = SendData() << 8;
        data |= SendData();
        Deselect();

        uint8_t config = GetConfigReg() & 0x80;
        SetConfigReg(0x20 | config);     // One-shot mode, Start conversion

        return data/128.0f;    
    }
}

