//--------------------------------------------------------------
//  Class for ADT7310 on one-shot mode (Header)
//      Default: 13-bit resolution
//  You want to know format of command byte for SPI,
//  see Table15 on data sheet of ADT7310, Rev. A p.18.
//
//  2017/10/28, Copyright (c) 2017 MIKAMI, Naoki
//--------------------------------------------------------------

#ifndef ADT7310_HPP
#define ADT7310_HPP

#include "mbed.h"

namespace Mikami
{
    class ADT7310
    {
    private:
        enum Reg { ST = 0x00, CONFIG = 0x08, DATA = 0x10 };
        DigitalOut ss_;     // Object for output of ss

    public:
        // Constructor
        ADT7310(PinName ss,             // for slave select
                PinName mosi = D11,     // MOSI
                PinName miso = D12,     // MISO
                PinName sck  = D13);    // SCK
        // Software reset
        void Reset();
        // Set resolution
        void SetResolution16(bool res = false);
        // Get specified register
        uint8_t GetReg(Reg myReg);
        // Get configuration register
        uint8_t GetConfigReg() { return GetReg(CONFIG); }
        // Get status register
        uint8_t GetStatusReg() { return GetReg(ST); }
        // Read data and translate to temperature
        float Read();

    protected:
        SPI spi_;           // Object of SPI

        // Select SPI device
        void Select() { ss_ = 0; }
        // Deselect SPI device
        void Deselect() { ss_ = 1; }
        // Set configuration register
        void SetConfigReg(uint8_t value);
        // Pre-transform for read
        void SendRead(Reg myReg) { spi_.write(myReg | 0x40); }
        // Pre-transform for write
        void SendWrite(Reg myReg) { spi_.write(myReg); }
        // Transform data
        uint8_t SendData(uint8_t data = 0) { return spi_.write(data); }
    };
}
#endif  // ADT7310_HPP

