//------------------------------------------------------
// Class for ADC using TIM2 trigger -- Header
//
// 2014/12/31, Copyright (c) 2014 MIKAMI, Naoki
//------------------------------------------------------

#ifndef ADC_BASE_HPP
#define ADC_BASE_HPP

#include "mbed.h"

namespace Mikami
{
    class ADC_Base
    {
    private:
        // Following object of AnalogIn class will be
        // initialized by menber initializer
        AnalogIn adc_;
        // Following two objects of AnalogIn class will be
        // initailized by regular executable statements
        AnalogIn* adc2_;
        AnalogIn* adc3_;

        // Channel of ADC1
        uint8_t ch1_, ch2_, ch3_;

        // Set timer to generate sampling pulse for ADC
        void SetTim2(int frequency);
        
        // Exctract channel number
        uint8_t GetChannelNumber(PinName);

        // for inhibition of copy constructor
        ADC_Base(const ADC_Base&);
        // for inhibition of substitute operator
        ADC_Base& operator=(const ADC_Base&);

    protected:
        // for normalize   
        static const float AMP_ = 1.0f/2048.0f;

        ADC_TypeDef* const myAdc_;

        // Wait until completion of AD conversion
        void WaitDone()
        { while((myAdc_->SR & ADC_SR_EOC) == RESET); }

    public:

        // Constructor
        //      pin1:       Pin Name for input as A0, A1, etc.
        //      frequency:  Sampling frequency
        //      pin2:       If use 2nd channel set this parameter
        //      pin3:       If use 3rd channel set this parameter
        ADC_Base(PinName pin1, int frequency,
                 PinName pin2 = NC, PinName pin3 = NC);  
        
        // Read ADC with waiting, range: [0, 0x0FFF]
        virtual uint16_t Read_u16()
        {
            WaitDone();
            return myAdc_->DR;   
        }

        // Read ADC with waiting, range: [-1.0f, 1.0f]
        virtual float Read()
        {
            WaitDone();
            return AMP_*((int16_t)myAdc_->DR - 2048);
        }

        // Select channel
        void Select1stChannel() { myAdc_->SQR3 = ch1_; }
        void Select2ndChannel() { myAdc_->SQR3 = ch2_; }
        void Select3rdChannel() { myAdc_->SQR3 = ch3_; }
        
        // Software start
        virtual void SoftStart()
        { myAdc_->CR2 |= ADC_CR2_SWSTART; }
        
        uint32_t ReadSQR3() { return myAdc_->SQR3; }
    };
}
#endif  // ADC_BASE_HPP
