//------------------------------------------------------
// Class for ADC using TIM2 trigger
//      To get bit definition for register in
//      peripheral, see "stm32f401xe.h"
//
// 2014/12/31, Copyright (c) 2014 MIKAMI, Naoki
//------------------------------------------------------

#include "ADC_Base.hpp"

namespace Mikami
{
    ADC_Base::ADC_Base(PinName pin1, int frequency,
                       PinName pin2, PinName pin3)
        : adc_(pin1), myAdc_(ADC1)
    {
        myAdc_->CR2 = ADC_EXTERNALTRIGCONVEDGE_RISING   // External Trigger on the rising edge
                    | ADC_EXTERNALTRIGCONV_T2_TRGO      // Use Timer2 TRGO event
                    | ADC_CR2_ADON;                     // Enable ADC

        ch1_ = GetChannelNumber(pin1);
        if (pin2 != NC)
        {
            adc2_ = new AnalogIn(pin2);
            ch2_ = GetChannelNumber(pin2);
        }
        if (pin3 != NC)
        {
            adc3_ = new AnalogIn(pin3);
            ch3_ = GetChannelNumber(pin3);
        }
        SetTim2(frequency);
        Select1stChannel();
    }

    // Extract channel number
    uint8_t ADC_Base::GetChannelNumber(PinName pin)
    {
        uint8_t ch = 0;
        if ((pin & 0x30) == 0x00) ch = pin;
        if ((pin & 0x30) == 0x10) ch = (pin & 0x01) + 8;
        if ((pin & 0x30) == 0x20) ch = (pin & 0x07) + 10;
        return ch;
    }

    void ADC_Base::SetTim2(int frequency)
    {
        __TIM2_CLK_ENABLE();    // Supply clock, See "stm32f4xx_hal_rcc.h"
        
        SystemCoreClockUpdate();    // Update core clock (for F411RE)
                                    // See system_stm32f4xx.h
        TIM_TypeDef* myTim = TIM2;

        myTim->CR2 = TIM_CR2_MMS_1; // Update event: as trigger out
        myTim->ARR = SystemCoreClock/frequency - 1; // Auto-reload
        myTim->PSC = 0;             // Prescaler
        myTim->CR1 = TIM_CR1_CEN;   // Enable TIM2
    }
}
