//-------------------------------------------------------
//  Class for LCD, ACM1602Ni
//
//  2016/04/01, Copyright (c) 2016 MIKAMI, Naoki
//-------------------------------------------------------

#include "ACM1602NI.hpp"

namespace Mikami
{
    // Constructor
    Acm1602Ni::Acm1602Ni(PinName sda, PinName scl, uint32_t clock,
                         bool cursor, bool blink)
        : i2c_(sda, scl), myI2c_((I2C_TypeDef*)NULL)
    {
        if ( ((sda == PB_9) || (sda == PB_7)) &&
             ((scl == PB_8) || (scl == PB_6)) )
                myI2c_ = (I2C_TypeDef*)I2C_1;   // I2C1 will be used
        if ( (sda == PB_3) && (scl == PB_10) )
                myI2c_ = (I2C_TypeDef*)I2C_2;   // I2C2 will be used
        if ( ((sda == PC_9) || (sda == PB_4)) &&
             (scl == PA_8) )
                myI2c_ = (I2C_TypeDef*)I2C_3;   // I2C3 will be used

        if (clock != 100000) i2c_.frequency(clock);

        wait_ms(40);
        connected_ = Clear();      // Clear display
        if (!connected_)
        {
            fprintf(stderr, "\r\nLCD ACM1602NI not connected\r\n");
            return;
        }

        WriteCmd(0x38); // data length：8-bit, 2-line, 5×8 dots
        WriteCmd(0x0C | (cursor << 1) | blink);
        WriteCmd(0x06); // cursor direction: rightward
    }

    // All clear
    bool Acm1602Ni::Clear()
    {
        bool ok = WriteCmd(0x01);
        wait_ms(50);
        return ok;
    }

    // Write string
    void Acm1602Ni::WriteString(const char str[])
    {
        for (int n=0; n<16; n++)
            if (str[n] == 0) break;
            else             WriteChar(str[n]);
    }

    // Write string from specified position
    void Acm1602Ni::WriteStringXY(const char str[],
                                  uint8_t x, uint8_t y)
    {
        SetXY(x, y);
        WriteString(str);
    }

    //---------------------------------------------------
    // Following member functions: private

    // Send command and data
    bool Acm1602Ni::LcdTx(uint8_t cmdData, uint8_t data)
    {
        // Generate start condition
        i2c_.start();

        // Send slave address
        TxDR(LCD_ADDRESS_);

        if (!IsTxMode()) return false;
        
        // Define kind of "data" in next statement
        TxDR(cmdData);
        TxDR(data);
        wait_us(500);   // indispensable
        
        // Generate stop condition
        i2c_.stop();      

        return true;
    }

    // Check on transmitter mode
    bool Acm1602Ni::IsTxMode()
    {
        for (int n=0; n<10; n++)
        {
            wait_us(10);      
            if (CheckSR12(I2C_SR1_TXE | I2C_SR1_ADDR,
                          I2C_SR2_MSL | I2C_SR2_BUSY
                                      | I2C_SR2_TRA))
                return true;
        }
        
        return false;
    }
}
