//------------------------------------------------------------------------------
//  "UITDSP_ADDA" に含まれるライブラリの使用例
//
//      アナログ入力
//          A0: ライン
//          A1: マイク
//          A2: 可変抵抗器
//
//  2015/02/23, Copyright (c) 2015 MIKAMI, Naoki
//------------------------------------------------------------------------------

#include "InternalADC.hpp"      //　かならず必要 

// 以下の "#define" 文を一つだけ有効にし，他はコメントとすること
#define POLLING_1_CHANNEL
//#define POLLING_SW_CHANNEL
//#define INTERRUPT_1_CHANNEL
//#define INTERRUPT_2_CHANNELS
//#define POLLING_DA_2_CHANNELS

using namespace Mikami;         //　かならず必要 

//------------------------------------------------------------------------------
// 基本的な使い方：ポーリング，１チャンネル

#ifdef POLLING_1_CHANNEL
#include "DAC_MCP4921.hpp"

const int TS_ = 100;    // 標本化間隔： 0.1 [ms]
InternalADC adc_(A0);   // A0 からの入力を使うように設定
DAC_MCP4921 myDac_;     // DAC を使うための設定

int main()
{
    printf("\r\nPOLLING_1_CHANNEL\r\n");
    
    adc_.SetSamplingPeriod(TS_);    // 標本化間隔の設定
    myDac_.ScfClockTim3(420000);    // 出力の LPF の遮断周波数を 4.2 kHz に設定

    while (true)
    {
        float xn = adc_.Read();     // A0 からの入力信号を読み込む
        //-----------------------------------------------
        // この部分に信号処理のプログラムを記述する
        
        float yn = xn;  // 処理を行わず，そのまま出力する
        //-----------------------------------------------
        myDac_.Write(yn);           // DAC へ出力する
    }
}
#endif  // POLLING_1_CHANNEL
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// ポーリングで入力の切り替えを行う場合

#ifdef POLLING_SW_CHANNEL
#include "DAC_MCP4921.hpp"

const int TS_ = 100;    // Sampling period: 0.1 ms
InternalADC adc_(A0, A1);   // for AD
DAC_MCP4921 myDac_;         // for DA

DigitalIn sw_(D2, PullDown);

int main()
{
    printf("\r\nPOLLING_SW_CHANNEL\r\n");

    adc_.SetSamplingPeriod(TS_);
    myDac_.ScfClockTim3(420000);    // cutoff frequency: 4.2 kHz

    int swBefore = 0;
    while (true)
    {
        float xn = adc_.Read();     // Read from A0 or A1
        //-----------------------------------------------
        float yn = xn;
        //-----------------------------------------------
        myDac_.Write(yn);           // Write to DAC

        int swNow = sw_.read();
        if (swNow != swBefore)
        {
            if (swNow == 0) adc_.Select1stChannel(); // line input
            else            adc_.Select2ndChannel(); // mic. input
            swBefore = swNow;
        }
    }
}
#endif  // POLLING_SW_CHANNEL
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// ADC の EOC 割り込みを使う場合，入力：１チャンネル

#ifdef INTERRUPT_1_CHANNEL
#include "ADC_Interrupt.hpp"    // for ADC not using ADC interrupt
#include "DAC_MCP4922.hpp"

const int TS_ = 100;    // Sampling period: 0.1 ms
ADC_Intr adc_(A0);  // for AD
DAC_MCP4922 myDac_; // for DA

// Interrupt service routine for ADC
void AdcIsr()
{   
    float xn = adc_.Read(); // Read from A0
    myDac_.Write(xn);       // to DAC
}

int main()
{
    printf("\r\nINTERRUPT_1_CHANNEL\r\n");

    adc_.SetSamplingPeriod(TS_);
    myDac_.ScfClockTim3(420000);    // cutoff frequency: 4.2 kHz

    adc_.SetIntrVec(AdcIsr);        // Assign ISR for ADC interrupt

    while (true) {}
}
#endif  // INTERRUPT_1_CHANNEL
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//　ADC の EOC 割り込みを使う場合で，２チャンネル使う場合

#ifdef INTERRUPT_2_CHANNELS
#include "ADC_Interrupt.hpp"    // for ADC not using ADC interrupt
#include "DAC_MCP4922.hpp"

//      A0: Line input
//      A1: Not used
//      A2: VR (controls volume of output signal)
const int TS_ = 100;    // Sampling period: 0.1 ms
ADC_Intr adc_(A0, A1, A2);   // for AD
DAC_MCP4922 myDac_(DAC_MCP4922::DAC_A, D11, D13, D10, D12);     // for DA

uint16_t a2_ = 0;
uint16_t a21_ = 0;
float vol_ = 1.0f;

// Interrupt service routine for ADC
void AdcIsr()
{   
    float xn = adc_.Read();     // Read from A0

    adc_.Select3rdChannel();    // Enable A2   
    adc_.SoftStartDisableIRQ(); // ADC start for A2 input
    
    float yn = vol_*xn; // Volume control by VR
    myDac_.Write(yn);   // to DAC

    // Read value which controls volume
    a2_ = adc_.ReadWait_u16();

    adc_.Select1stChannel();        // Enable A0
    adc_.ClearPending_EnableIRQ();  // Clear pending interrupt
}

int main()
{
    printf("\r\nINTERRUPT_2_CHANNELS\r\n");

    adc_.SetSamplingPeriod(TS_);
    myDac_.ScfClockTim3(420000);    // cutoff frequency: 4.2 kHz

    adc_.SetIntrVec(AdcIsr);        // Assign ISR for ADC interrupt

    while (true)
    {
        if (abs(a2_ - a21_) > 10)
        {
            vol_ = ((float)a2_)/4095.0f;
            a21_ = a2_;
        }
        wait(0.1f);
    }
}
#endif  // INTERRUPT_2_CHANNEL
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// DAC を２チャンネル出力で使う場合, 入力：ポーリング，１チャンネル

#ifdef POLLING_DA_2_CHANNELS
#include "DAC_MCP4922Dual.hpp"

InternalADC adc_(A0);   // for AD
DAC_MCP4922Dual myDac_; // for DA

int main()
{
    printf("\r\nPOLLING_DA_2_CHANNELS\r\n");
    adc_.SetSamplingPeriod(20); // 20 micro seconds

    while (true)
    {
        float xn = adc_.Read();     // Read from A0
        //-----------------------------------------------
        float yn = xn;
        //-----------------------------------------------
        myDac_.Write(yn, yn);       // Write to DAC
    }
}
#endif  // POLLING_DA_2_CHANNEL
//------------------------------------------------------------------------------


