//------------------------------------------------------------------------------
//  "UITDSP_ADDA" に含まれるライブラリの使用例
//
//      アナログ入力
//          A0: ライン
//          A1: マイク
//          A2: 可変抵抗器
//
//  2015/07/25, Copyright (c) 2015 MIKAMI, Naoki
//------------------------------------------------------------------------------

#include "ADC_BuiltIn.hpp"      // かならず必要

// 以下の "#define" 文を一つだけ有効にし，他はコメントとすること
#define POLLING_1_CHANNEL
//#define POLLING_SW_CHANNEL
//#define INTERRUPT_1_CHANNEL
//#define INTERRUPT_2_CHANNELS
//#define POLLING_DA_2_CHANNELS

using namespace Mikami;         //　かならず必要

//------------------------------------------------------------------------------
// 基本的な使い方：ポーリング，１チャンネル

#ifdef POLLING_1_CHANNEL
#include "DAC_MCP4922.hpp"      // DAC MCP4922 用

const int FS_ = 10000;          // 標本化周波数: 10 kHz
ADC_BuiltIn adc_(A0, FS_);      // A0 からの入力を使うように設定
DAC_MCP4922 myDac_;             // DAC を使うための設定

int main()
{
    printf("\r\nPOLLING_1_CHANNEL\r\n");
    myDac_.ScfClockTim3(420000);    // 出力の LPF の遮断周波数を 4.2 kHz に設定

    while (true)
    {
        float xn = adc_.Read();     // A0 からの入力信号を読み込む
        //-----------------------------------------------
        // この部分に信号処理のプログラムを記述する
        
        float yn = xn;  // 処理を行わず，そのまま出力する
        //-----------------------------------------------
        myDac_.Write(yn);           // DAC へ出力する
    }
}
#endif  // POLLING_1_CHANNEL
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// ポーリングで入力の切り替えを行う場合

#ifdef POLLING_SW_CHANNEL
#include "DAC_MCP4922.hpp"

const int FS_ = 10000;          // 標本化周波数: 10 kHz
ADC_BuiltIn adc_(A0, FS_, A1);  // ADC の A0, A1 を使うための設定
DAC_MCP4921 myDac_;             // DAC を使うための設定

DigitalIn sw_(D2, PullDown);

int main()
{
    printf("\r\nPOLLING_SW_CHANNEL\r\n");
    myDac_.ScfClockTim3(420000);    // 出力の LPF の遮断周波数を 4.2 kHz に設定する

    int swBefore = 0;
    while (true)
    {
        float xn = adc_.Read();     // A0 または A1 からの入力信号を読み込む
        //-----------------------------------------------
        float yn = xn;
        //-----------------------------------------------
        myDac_.Write(yn);           // DAC へ出力する

        int swNow = sw_.read();
        if (swNow != swBefore)
        {
            if (swNow == 0) adc_.Select1stChannel(); // 入力：　ライン
            else            adc_.Select2ndChannel(); // 入力：　マイク
            swBefore = swNow;
        }
    }
}
#endif  // POLLING_SW_CHANNEL
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// ADC の EOC 割り込みを使う場合，入力：１チャンネル

#ifdef INTERRUPT_1_CHANNEL
#include "ADC_Interrupt.hpp"    // ADC 変換終了割り込みを使う場合
#include "DAC_MCP4922.hpp"

const int FS_ = 10000;  // 標本化周波数: 10 kHz
ADC_Intr adc_(A0, FS_); // ADC の A0 を使うための設定
DAC_MCP4922 myDac_;     // DAC を使うための設定

// ADC 変換終了割り込みに対する割り込みサービス･ルーチン
void AdcIsr()
{   
    float xn = adc_.Read(); // A0 からの入力信号を読み込む
    myDac_.Write(xn);       // DAC へ出力する
}

int main()
{
    printf("\r\nINTERRUPT_1_CHANNEL\r\n");

    myDac_.ScfClockTim3(420000);    // 出力の LPF の遮断周波数を 4.2 kHz に設定する

    // ADC 変換終了割り込みに対する割り込みサービス･ルーチンを割り当てる
    adc_.SetIntrVec(AdcIsr);        // Assign ISR for ADC interrupt

    while (true) {}
}
#endif  // INTERRUPT_1_CHANNEL
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//　ADC の EOC 割り込みを使う場合で，２チャンネル使う場合
//      A0 から読み込んだ信号の振幅を，A2 に接続されている半固定抵抗器で，
//      コントロールして出力する

#ifdef INTERRUPT_2_CHANNELS
#include "ADC_Interrupt.hpp"    // ADC 変換終了割り込みを使う場合
#include "DAC_MCP4922.hpp"

const int FS_ = 10000;          // 標本化周波数: 10 kHz
ADC_Intr adc_(A0, FS_, A2);     // ADC の A0, A2 を使うための設定
DAC_MCP4922 myDac_;             // DAC を使うための設定

uint16_t vr_ = 0;   // A2 から読み込んだ現在の値
uint16_t vr1_ = 0;  // A2 から，一つ前に読み込んだ値
float vol_ = 1.0f;  // 出力の振幅の倍率

// ADC 変換終了割り込みに対する割り込みサービス･ルーチン
void AdcIsr()
{   
    float xn = adc_.Read();     // A0 からの入力信号を読み込む

    adc_.Select2ndChannel();    // 入力チャンネルとして A2 を選択する
    adc_.SoftStart();           // AD 変換を開始する（ADC 変換終割り込みは禁止する）
    
    float yn = vol_*xn; // 出力信号の振幅を半固定抵抗器の値でコントロールする
    myDac_.Write(yn);   // DAC へ出力する

    // 半固定抵抗器の値を A2 から読み込む
    vr_ = adc_.ReadWait_u16();

    adc_.Select1stChannel();        // 入力チャンネルとして A0 を選択する
    adc_.ClearPending_EnableIRQ();  // 保留中の割り込みをクリアする
}

int main()
{
    printf("\r\nINTERRUPT_2_CHANNELS\r\n");
    myDac_.ScfClockTim3(420000);    // 出力の LPF の遮断周波数を 4.2 kHz に設定する

    // ADC 変換終了割り込みに対する割り込みサービス･ルーチンを割り当てる
    adc_.SetIntrVec(AdcIsr);

    while (true)
    {
        if (abs(vr_ - vr1_) > 10)
        {
            vol_ = ((float)vr_)/4095.0f;
            vr1_ = vr_;
        }
        wait(0.1f);
    }
}
#endif  // INTERRUPT_2_CHANNEL
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// DAC を２チャンネル出力で使う場合, 入力：ポーリング，１チャンネル

#ifdef POLLING_DA_2_CHANNELS
#include "DAC_MCP4922Dual.hpp"

const int FS_ = 10000;      // 標本化周波数: 10 kHz
ADC_BuiltIn adc_(A0, FS_);  // ADC の A0 を使うための設定
DAC_MCP4922Dual myDac_;     // DAC の２つのチャンネルを使うための設定

int main()
{
    printf("\r\nPOLLING_DA_2_CHANNELS\r\n");

    while (true)
    {
        float xn = adc_.Read();     // A0 からの入力信号を読み込む
        //-----------------------------------------------
        float yn = xn;
        //-----------------------------------------------
        myDac_.Write(yn, -yn);      // DAC の２つのチャンネルへ出力する
    }
}
#endif  // POLLING_DA_2_CHANNEL
//------------------------------------------------------------------------------
