//--------------------------------------------------------------
//  Class for DAC MCP4921 -- Header
//  This class can be also used for MCP4922 (only channel A)
//
//  Default pin assign
//      D11  SPI Master Out Slave In
//      D13  SPI Serial Clock
//      D10  SPI Slave Select
//      D12  to MCP4921 LDAC pin
//      D9   clock for Switched-capacitor filter ---- TIM3
//
//  Argument "cs" in constructor can be used for any pin
//  corresponding to DigitalOut class in mbed
//
//  Argument "pin" in member function ScfClockTim3() can be set
//  PA_6(D12), PB_4(D5), PC_6, PB_5(D4), PC_7(D9),
//  PC_8, or PC_9
//
//  2015/01/31, Copyright (c) 2015 MIKAMI, Naoki
//--------------------------------------------------------------

#ifndef DAC_MCP4921_HPP
#define DAC_MCP4921_HPP

#include "mbed.h"

namespace Mikami
{
    class DAC_MCP4921
    {
    public:
        // Constructor
        DAC_MCP4921(
            PinName mosi = SPI_MOSI,    // D11
            PinName sclk = SPI_SCK,     // D13
            PinName cs   = SPI_CS,      // D10
            PinName ldac = SPI_MISO);   // D12

        // -1.0f <= value <= 1.0f
        void Write(float value)
        {
            if (value < -1.0f) value = -1.0f;
            if (value >  1.0f) value =  1.0f;
            WriteDac((uint16_t)((value + 1.0f)*2047));
        }

        // 0 <= value <= 4095
        void Write(uint16_t value)
        {   WriteDac((value > 4095) ? 4095 : value); }
        
        // generate LDAC negative-going pulse
        void Ldac()
        {
            ld_.write(0);
            ld_.write(0);   // ensure width of "L" pulse
            ld_.write(1);
        }

        // Check busy
        bool IsBusy()
        {   return (mySpi_->SR & SPI_FLAG_BSY) == SPI_FLAG_BSY; }
        
        // Little wait
        void Wait()
        {   __NOP(); __NOP(); __NOP();  __NOP(); __NOP(); }

        // Set clock for switched-capacitor filter
        void ScfClockTim3(uint32_t clock, PinName pin = D9);

    protected:
        void SlaveSelect() { *cs_ = 0; }
        void WriteSpi(uint16_t value) { mySpi_->DR = value; }

    private:
        SPI spi_;       // SPI object of mbed
        DigitalOut ld_; // for LDAC

        static DigitalOut* cs_;
        // Pointer of I2C
        static SPI_TypeDef* mySpi_;
       
        // for inhibition of copy constructor
        DAC_MCP4921(const DAC_MCP4921&);
        // for inhibition of substitute operator
        DAC_MCP4921& operator=(const DAC_MCP4921&);     
        
        // for internal use
        virtual void WriteDac(uint16_t value)
        {
            while (IsBusy()) {}
            SlaveSelect();
            WriteSpi(value | 0x3000);
        }
        
        // ISR for RXNE interrupt of SPI
        static void Isr()
        {
            *cs_ = 1;
            uint16_t value = mySpi_->DR;    // Clear RXNE
        }
    };
}
#endif  // DAC_MCP4921_HPP

