//------------------------------------------------------
// Class for ADC using TIM2 trigger -- Header
//
// 2015/04/19, Copyright (c) 2015 MIKAMI, Naoki
//------------------------------------------------------

#ifndef ADC_BUILTIN_HPP
#define ADC_BUILTIN_HPP

#include "mbed.h"

namespace Mikami
{
    class ADC_BuiltIn
    {
    private:
        // Following object of AnalogIn class will be
        // initialized by menber initializer
        AnalogIn adc_;
        // Following two objects of AnalogIn class will be
        // initailized by regular executable statements
        AnalogIn* adc2_;
        AnalogIn* adc3_;

        // Channel of ADC1
        uint8_t ch1_, ch2_, ch3_;

        // Set timer to generate sampling pulse for ADC
        void SetTim2(int frequency);
        
        // Exctract channel number
        uint8_t GetChannelNumber(PinName);

        // for inhibition of copy constructor
        ADC_BuiltIn(const ADC_BuiltIn&);
        // for inhibition of substitute operator
        ADC_BuiltIn& operator=(const ADC_BuiltIn&);

    protected:
        // for normalize   
        static const float AMP_ = 1.0f/2048.0f;

        ADC_TypeDef* const myAdc_;

        // Wait until completion of AD conversion
        void WaitDone()
        { while((myAdc_->SR & ADC_SR_EOC) == RESET); }

    public:

        // Constructor
        //      pin1:       Pin Name for input as A0, A1, etc.
        //      frequency:  Sampling frequency
        //      pin2:       If use 2nd channel, set this parameter
        //      pin3:       If use 3rd channel, set this parameter
        ADC_BuiltIn(PinName pin1, int frequency,
                 PinName pin2 = NC, PinName pin3 = NC);  
        
        // Read ADC with waiting, range: [0, 0x0FFF]
        virtual uint16_t Read_u16()
        {
            WaitDone();
            return myAdc_->DR;   
        }

        // Read ADC with waiting, range: [-1.0f, 1.0f]
        virtual float Read()
        {
            WaitDone();
            return AMP_*((int16_t)myAdc_->DR - 2048);
        }

        // Select channel
        void Select1stChannel() { myAdc_->SQR3 = ch1_; }
        void Select2ndChannel() { myAdc_->SQR3 = ch2_; }
        void Select3rdChannel() { myAdc_->SQR3 = ch3_; }
    };
}
#endif  // ADC_BUILTIN_HPP
