//------------------------------------------------------
// Class for single DAC in MCP4921 -- Header
//      Fast version
//
// Default pin assign
//      D11  SPI Master Out Slave In
//      D13  SPI Serial Clock
//      D10  SPI Slave Select ----------------------- TIM4
//      D12  to MCP4921 LDAC pin
//      D9   clock for Switched-capacitor filter ---- TIM3
//
// Argument cs in constructor must be output of TIM4,
// i.e. D10(PB_6), PB_7, D15(PB_8), or D14(PB_9)
//
// Argument pin in function ScfClockTim3() can be
// PA_6(D12), PB_4(D5), PC_6, PB_5(D4), PC_7(D9),
// PC_8, or PC_9
//
// 2015/03/27, Copyright (c) 2015 MIKAMI, Naoki
//------------------------------------------------------

#ifndef DAC_MCP4921_HPP
#define DAC_MCP4921_HPP

#include "mbed.h"
#include "TIM4_SlaveSelect.hpp"

namespace Mikami
{
    class DAC_MCP4921
    {
    public:
        // Constructor
        DAC_MCP4921(
            PinName mosi = SPI_MOSI,    // D11
            PinName sclk = SPI_SCK,     // D13
            PinName cs   = SPI_CS,      // D10
            PinName ldac = SPI_MISO);   // D12

        // -1.0f <= value <= 1.0f
        void Write(float value)
        {
            if (value < -1.0f) value = -1.0f;
            if (value >  1.0f) value =  1.0f;

            WriteDac((value + 1.0f)*2047);
        }

        // 0 <= value <= 4095
        void Write(uint16_t value)
        {   WriteDac((value > 4095) ? 4095 : value); }

        // Check busy
        bool IsBusy()
        {   return (mySpi_->SR & SPI_SR_BSY) == SPI_SR_BSY; }

        // Set clock for switched-capacitor filter
        void ScfClockTim3(uint32_t clock, PinName pin = D9);

    protected:
        void SlaveSelect() { ss_->SlaveSelect(); }
        void WriteSpi(uint16_t value) { mySpi_->DR = value; }
        
        // generate LDAC negative-going pulse
        void Ldac()
        {
            ld_.write(0);
            ld_.write(0);   // ensure width of "L" pulse
            ld_.write(1);
        }

    private:
        SPI spi_;               // SPI object of mbed
        TIM4_SlaveSelect* ss_;  // Slave Select for DAC
        DigitalOut ld_;         // for LDAC

        // Pointer for SPI
        SPI_TypeDef* mySpi_;

        // for inhibition of copy constructor
        DAC_MCP4921(const DAC_MCP4921&);
        // for inhibition of substitute operator
        DAC_MCP4921& operator=(const DAC_MCP4921&);     
        
        // for internal use
        virtual void WriteDac(uint16_t value)
        {
            while (IsBusy()) {}
            SlaveSelect();
            WriteSpi(value | 0x3000);
        }       
    };
}
#endif  // DAC_MCP4921_HPP
