//--------------------------------------------------------------
// Comb filter
// 2014/10/23, Copyright (c) 2014 MIKAMI, Naoki
//--------------------------------------------------------------

#ifndef REVERB_UNIT_HPP
#define REVERB_UNIT_HPP

#include "mbed.h"

// Base class for reverb unit
template<int delay> class Reverb
{
private:
    int ptr_;
    float un_[delay];    // for delay
protected:
    float Get() { return un_[ptr_]; }
    void Set(float x)
    {
        un_[ptr_] = x;   
        if (++ptr_ >=  delay) ptr_ = 0;
    }
public:
    // Constructor
    Reverb() : ptr_(0) { Clear(); }

    // Execute of filter (pure virtual function)
    virtual float Execute(float x) = 0;

    // Clear internal delay elements
    void Clear()
    { for (int n=0; n<delay; n++) un_[n] = 0; }
};    
   
// Comb filter
template<int delay> class CombFilter : public Reverb<delay>
{
private:
    const float GC_;
public:
    // Constructor
    CombFilter(float g) : GC_(g) {}

    // Execute comb filter 
    virtual float Execute(float x)
    {
        float yn = Reverb<delay>::Get();
        Reverb<delay>::Set(x + GC_*yn);
        return yn;
     }
};

// Allpass filter
template<int delay> class AllpassFilter : public Reverb<delay>
{
private:
    const float G0_;
public:
    // Constructor
    AllpassFilter(float g) : G0_(g) {}

    // Execute comb filter 
    virtual float Execute(float x)
    {
        float un = x + G0_*Reverb<delay>::Get();
        float yn = -G0_*un + Reverb<delay>::Get();
        Reverb<delay>::Set(un);
        return yn;
     }
};

#endif  // REVERB_UNIT_HPP

