//---------------------------------------------------------------------
//  FFT によるスペクトルアナライザ，白色雑音発生器付き (Nucleo-F446RE 用)
//      ● PC 側のプログラム： "CQ_FFT_Analyzer"
//      ● ボーレート
//          ポートの検索時：   9,600 baud
//          ポートの検索後： 460,800 baud
//
//      ● 入力：  A1
//      ● 白色雑音の出力：A2
//
//  2021/10/22, Copyright (c) 2021 MIKAMI, Naoki
//---------------------------------------------------------------------

#include <string>
#include "Array.hpp"
#include "DSP_AdcIntr.hpp"
#include "DSP_Dac.hpp"
#include "FFT_Analyzer.hpp"
#include "DoubleBuffer.hpp"
#include "Coefs_IIR_LP.hpp" // 縦続形 IIR フィルタの係数
#include "IirCascade.hpp"   // 縦続形 IIR フィルタ
#include "MSeq16.hpp"       // M 系列発生器
#include "XferSpectrum.hpp"
using namespace Mikami;

#ifndef __STM32F446xx_H
#error "Use Nucleo-F446RE"
#endif

const int N_FFT_ = 1024;            // FFT の点数
const int N_FRAME_ = N_FFT_;        // １フレーム当たり標本化するデータ数
const int N_SPC_ = N_FFT_/2 + 1;    // 有効なスペクトルの点数
const int RATIO_ = 10;              // ダウンサンプリングの倍率：1/10
const int N_TX_ = 501;              // PC に転送するデータ数

DoubleBuffer buf_(N_FRAME_);            // ダウンサンプリングの結果を保存するバッファ
DspAdcIntr myAdc_(10.24f*RATIO_, A1);   // 標本化周波数： 102.4 kHz
DspDac myDac_;
IirCascade df1_(ORDER1_, CK1_, G01_);   // ダウンサンプリング用 Anti-alias フィルタ
IirCascade df2_(ORDER2_, CK2_, G02_);   // 白色雑音発生用低域通過フィルタ
MSeq16 mSeq_;                           // M 系列信号発生器（N = 16）

// ADC 変換終了割り込みに対する割り込みサービス･ルーチン
void AdcIsr()
{
    static int count = 0;

    float xn = myAdc_.Read();   // AD 変換された値を取得

    float noise = df2_.Execute(mSeq_.Execute());
    myDac_.Write(1.6f*noise);   // 白色雑音出力

    float yn = df1_.Execute(xn);    // ダウンサンプリング用 Anti-alias フィルタの実行

    if (++count >= RATIO_)
    {
        buf_.Store(yn);         // ダウンサンプリングされたデータをバッファへ格納
        count = 0;
        buf_.IsFullSwitch();    // バッファが満杯であればバッファを切り替える
    }
}

int main()
{
    SerialRxTxIntr rxTx;                // PC との通信用，最初は 9600 baud
    XferSpectrum tx(rxTx, N_TX_);       // PC に転送するためのオブジェクトの生成
    FftAnalyzer analyzer(N_FFT_);       // FFT によるスペクトル解析オブジェクトの生成

    Array<float> sn(N_FFT_, 0.0f);      // スペクトル解析の対象となるデータ
    Array<float> absFt(N_SPC_);         // 解析結果：スペクトルの絶対値

    NVIC_SetPriority(ADC_IRQn, 0);      // AD変換終了割り込みの優先度が最高
    NVIC_SetPriority(USART2_IRQn, 1);

    bool ready = false; // スペクトルの計算終了で true
    bool okGo = false;  // "GO" を受信したら true

    myAdc_.SetIntrVec(&AdcIsr); // AD変換終了割り込みの割り当て
    while (true)
    {
        // PC からのコマンドの解析
        if (rxTx.IsEol())      // 受信バッファのデータが有効になった場合の処理
        {
            string str = rxTx.GetBuffer();
            if (str == "FFTAnalyzer")
            {
                rxTx.TxString("ACK\n"); // PC からの "FFTAnalyzer" に対して "ACK" を送信
                wait_ms(10);
                rxTx.Baud(460800);      // 以降は 460,800 baud
            }
            if (str.substr(0, 2) == "GO")
                okGo = true;  // データの転送要求あり
        }

        if (buf_.IsFull())  // 入力データが満杯の場合，以下の処理を行う
        {
            for (int n=0; n<N_FRAME_; n++) sn[n] = buf_.Get(n);
            analyzer.Execute(sn, absFt);    // スペクトル解析の実行
            ready = true;                   // スペクトル解析終了
        }

        // 転送要求がありスペクトル解析が終了している場合にデータを PC へ転送する
        if (okGo && ready)
        {
            tx.ToPC(absFt); // データを PC へ転送
            ready = false;
            okGo = false;
        }
    }
}