//-------------------------------------------------------------
//  ソフトウェア割込み（SWI）を管理するクラス
//      登録数は 32 まで
//
//  2020/04/24, Copyright (c) 2020 MIKAMI, Naoki
//-------------------------------------------------------------

#include "mbed.h"

#ifndef SWI_MANAGER_HPP
#define SWI_MANAGER_HPP

namespace Mikami
{
    class SwiManager
    {
    public:
        // コンストラクタ
        //      IRQn      割込み番号
        //      vector    割込みサービス･ルーチン（ISR）の開始アドレス
        //      priority  割込み優先順位
        SwiManager(IRQn_Type IRQn, uint32_t vector, uint32_t priority = 0)
            : irqN_(IRQn), priority_(priority)
        {
            MBED_ASSERT(count_ <= TBL_SIZE_);   // 登録数オーバーのチェック

            if (count_ == 0)
                for (int n=0; n<TBL_SIZE_; n++) table_[n] = (IRQn_Type)255;

            // 同じ割込み要因が SWI 用に登録済みでないことをチェックする
            for (int n=0; n<TBL_SIZE_; n++)
                MBED_ASSERT(!(irqN_ == table_[n]));

            table_[count_++] = irqN_;       // 登録

            NVIC_SetVector(irqN_, vector);      // ISR の開始アドレスの設定
            NVIC_SetPriority(irqN_, priority_); // 優先順位の設定
        }

        // ソフトウェア割込み発生
        void Invoke() { NVIC->STIR = irqN_; }
        // ソフトウェア割込みを有効にする
        void Enable() { NVIC_EnableIRQ(irqN_); }
        // ソフトウェア割込みを無効にする
        void Disable() { NVIC_DisableIRQ(irqN_); }
        // 割込み優先順位を取得する
        uint32_t GetPriority() const { return priority_; }

    private:
        const static int TBL_SIZE_ = 32;    // 登録数の最大値
        static IRQn_Type table_[];          // 登録用の表
        static int count_;                  // 登録数

        IRQn_Type irqN_;    // 割込み要因
        uint32_t priority_; // 優先順位

        // コピー･コンストラクタおよび代入演算子の禁止のため
        SwiManager(const SwiManager& );
        SwiManager& operator=(const SwiManager& );
    };

    // static メンバの定義
    IRQn_Type SwiManager::table_[TBL_SIZE_];
    int SwiManager::count_ = 0;
}
#endif  // SWI_MANAGER_HPP