//-------------------------------------------------------------
//  Nucleo ボードの USB コネクタに対応するシリアルポート受送信割込みで
//  複数のキャラクタ受送信をサポ－トするクラス
//
//  "\r" を受信した場合に，複数キャラクタの受信の終了とする
//  キャラクタの数はデフォルトで 32 まで
//
//  ボーレートのデフォルト値は 9,600 boud
//  Nucleo-F446RE の場合，ファームウェアの更新をしなくても
//  115,200 boud までは動作可能
//
//  2021/04/12  メンバ関数 Baud() を追加
//
//  2021/04/12, Copyright (c) 2020 MIKAMI, Naoki
//-------------------------------------------------------------

#include "mbed.h"
#include <string>
#include "SerialTxIntr.hpp"

#ifndef SERIAL_RXTX_INTR_HPP
#define SERIAL_RXTX_INTR_HPP

namespace Mikami
{
    class SerialRxTxIntr
    {
    public:
        SerialRxTxIntr(int maxChar = 32, int baud = 9600);  // コンストラクタ
        bool IsEol();   // メッセージの終了であればそのメッセージを保存し true を返す
        string GetBuffer() { return buffer2_; }             // バッファの内容を取得
        void EchobackEnable() { echoOn_ = true; }           // エコーバック ON
        void EchobackDisable() { echoOn_ = false; }         // エコーバック OFF
        void TxString(string str) { tx_.TxString(str); }    // 文字列の送信
        void Baud(int baudRate) { pc_.baud(baudRate); }     // ボーレートの設定

    private:
        Serial pc_;
        SerialTxIntr tx_;

        const int RX_MAX_;  // 受信バッファのサイズ
        __IO bool eol_;     // "\r" を受信した場合に true
        string rxBuffer_;   // 受信バッファ
        string buffer2_;    // 受信バッファの内容を保存しておくバッファ
        bool echoOn_;       // エコーバックの有無

        // シリアル･ポートの受信割り込みに対する割り込みサービス･ルーチン
        void RxIsr();

        // コピー･コンストラクタ，代入演算子の禁止のため
        SerialRxTxIntr(const SerialRxTxIntr&);
        SerialRxTxIntr& operator=(const SerialRxTxIntr&);
    };
}
#endif  // SERIAL_RXTX_INTR_HPP