//--------------------------------------------------------------
//  SD_WavReader class
//      SD カードの *.wav ファイルの内容を読み出す
//      以下の型式のファイルのみ扱う
//          PCM，16 ビットステレオ，標本化周波数 44.1 kHz
//
//  2017/03/22, Copyright (c) 2017 MIKAMI, Naoki
//--------------------------------------------------------------

#include "SD_WavReader.hpp"

namespace Mikami
{
    SD_WavReader::SD_WavReader(int32_t bufferSize)
        : STR_("sd"), ok_(false)
    {
        sd_ = new SDFileSystem(STR_.c_str());
        sd_->mount();      
        buffer.SetSize(bufferSize*2);
    }

    SD_WavReader::~SD_WavReader()
    {
        sd_->unmount();
        delete sd_;
    }
    
    void SD_WavReader::Open(const string fileName)
    {
        string name = (string)"/" + STR_ + "/" + fileName;
        fp_ = fopen(name.c_str(), "rb");
        if (fp_ == NULL) ErrorMsg("open error!!");
    }
    
    
    // ファイルのヘッダ "RIFFxxxxWAVEfmt " の部分の読み込み
    //      戻り値：16 ビットステレオ，標本化周波数が 44.1 kHz の場合 true
    bool SD_WavReader::IsWavFile()
    {
        char data[17];
        fread(data, 1, 16, fp_);    // 16 バイト分を読み込む
        string strRead = "";
        for (int n=0; n<4; n++) strRead += data[n];
        // 間の 4 文字はチェックしない
        for (int n=8; n<16; n++) strRead += data[n];

        // "RIFF", "WAVE", "fmt " が存在することを確認
        if (strRead != "RIFFWAVEfmt ") return false;

        // fmt chunck のサイズを取得
        uint32_t fmtChunkSize;
        fread(&fmtChunkSize, sizeof(uint32_t), 1, fp_);

        // PCM, Stereo, 44.1 kHz, 16 bit であることを確認
        WaveFormatEx fmtData;
        fread(&fmtData, fmtChunkSize, 1, fp_);
        if ((fmtData.wFormatTag     != 1)   ||
            (fmtData.nChannels      != 2)   ||
            (fmtData.nSamplesPerSec != AUDIO_FREQUENCY_44K) ||
            (fmtData.wBitsPerSample != 16)
           ) return false;

        // data chunk を探す
        char dataId[5];
        dataId[4] = 0;
        fread(dataId, 1, 4, fp_);
        if ("data" != (string)dataId)
            for (int n=0; n<100; n++)
            {
                char oneByte;
                fread(&oneByte, 1, 1, fp_);
                for (int k=0; k<3; k++)
                    dataId[k] = dataId[k+1];
                dataId[3] = oneByte;
                if ("data" == (string)dataId) break;
                
                if (n == 99) return false;
            }

        // データサイズ (byte) を取得
        int32_t sizeByte;
        fread(&sizeByte, sizeof(int32_t), 1, fp_);
        size_ = sizeByte/4;

        ok_ = true;
        return true;
    }

    // ファイルからステレオデータの取得
    void SD_WavReader::ReadStereo(Array<int16_t>& dataL,
                                  Array<int16_t>& dataR)
    {
        if (!ok_) ErrorMsg("Get data FAILED");
        uint32_t size = dataL.Length();
        fread(buffer, sizeof(int16_t), size*2, fp_);       
        for (int n=0; n<size; n++)
        {
            dataL[n] = buffer[2*n];
            dataR[n] = buffer[2*n+1];
        }
    }

    // ファイルからデータをモノラルに変換しての取得
    void SD_WavReader::ReadAndToMono(Array<int16_t>& data)
    {
        if (!ok_) ErrorMsg("Get data FAILED");
        uint32_t size = data.Length();
        fread(buffer, sizeof(int16_t), size*2, fp_);
        for (int n=0; n<size; n++)
            data[n] = (buffer[2*n] + buffer[2*n+1])/2;
    }        
    
    // データサイズ（標本化点の数）の取得
    int32_t SD_WavReader::GetSize()
    {
        if (!ok_) ErrorMsg("Get data size FAILED");
        return size_;
    }
}
