//--------------------------------------------------------------
//  SD プレーヤー用抽象基底クラス
//
//  2017/03/22, Copyright (c) 2017 MIKAMI, Naoki
//--------------------------------------------------------------

#include "SD_PlayerSkeleton.hpp"

namespace Mikami
{
    SD_PlayerSkeleton::SD_PlayerSkeleton(string str, bool resetButton)
        : BUFF_SIZE_(2048), sn_(BUFF_SIZE_),
          mySai_(SaiIO::OUTPUT, BUFF_SIZE_, AUDIO_FREQUENCY_44K),
          sdReader_(BUFF_SIZE_),
          fileName_(""),
          title_(214, 4, str, Label::CENTER, Font16),
          selector_(0, 22, 256, 37, sdReader_),
          menu_(BG_LEFT_, 2, BG_WIDTH_, BG_HEIGHT_, 5,
                (string[]){"SELECT", "PLAY", "PAUSE", "RESUME", "STOP"},
                0, 2, 1)
    {
        menu_.InactivateAll();
        menu_.Activate(0);

        if (!selector_.CreateTable())
            BlinkLabel errLabel(240, 100, "SD CARD ERROR", Label::CENTER);

        if (resetButton) reset_ = new ResetButton();
        else             reset_ = NULL;
    }

    // SD プレーヤーの処理の実行
    void SD_PlayerSkeleton::Execute()
    {
        bool playOk = false;
        int32_t loopCount;

        while (true)
        {
            if (playOk)             // PAUSE 後の PLAY の場合
                loopCount = SD_Open();
            else                    // PAUSE 後の PLAY ではない場合
            {
                if (0 == WaitTouched()) SelectFile();
                loopCount = SD_Open();
                while (1 != WaitTouched()) {}   // PLAY がタッチされるまで待つ
            }

            Display();  // アプリ固有の表示処理（仮想関数）
            if (reset_ != NULL) reset_->Draw();

            menu_.Inactivate(0);    // SELECT ボタン無効
            menu_.Activate(2);      // PAUSE ボタン有効
            menu_.Activate(4);      // STOP ボタン有効

            playOk = false;
            bool stopOk = false;

            Clear();    // アプリ固有のクリア処理（仮想関数）

            mySai_.PlayOut();       // Play 開始

            // ファイルのデータがなくなるまで繰り返す
            for (int k=0; k<loopCount; k++)
            {
                int touch42 = -1;
                menu_.GetTouchedNumber(touch42);
                if (touch42 == 4) break;    // STOP
                if (touch42 == 2)           // PAUSE ボタンがタッチされた場合の処理
                {
                    menu_.Inactivate(2);    // PAUSE ボタン無効
                    menu_.Activate(3);      // RESUME ボタン有効
                    mySai_.PauseOut();

                    // PLAY か RESUME か STOP がタッチされるまで待つ
                    switch (WaitTouched())
                    {
                        case 1: playOk = true;      // 最初から PLAY
                                break;
                        case 3: mySai_.ResumeOut(); // PAUSE 箇所から PLAY 再開
                                menu_.Activate(2);
                                menu_.Inactivate(3);
                                menu_.TouchedColor(1);
                                break;
                        case 4: stopOk = true;      // STOP
                                break;
                    }
                }
                if (playOk || stopOk) break;

                DoIfHandled();
                // アプリ固有の１ブロック分に対する信号処理（純粋仮想関数）
                SignalProcessing();
            }
            mySai_.StopOut();
            if (!playOk) menu_.Activate(0); // SELECT ボタン有効
            menu_.Draw(1);                  // PLAY ボタンの色を初期状態にする
            for (int n=2; n<5; n++)         // その他のボタンは無効
                menu_.Inactivate(n);

            sdReader_.Close();   // SD のファイルのクローズ
        }
    }

    // 選択されたファイル名から拡張子を除いたものを取得
    string SD_PlayerSkeleton::GetFileNameNoExt()
    {
        string fName = fileName_;
        int k = fName.rfind(".");
        if (k != string::npos)
            return fName.erase(k);
        else
            return fName;
    }

    // SD カードのファイルのオープン
    int32_t SD_PlayerSkeleton::SD_Open()
    {
        if (fileName_.empty()) SelectFile();
        sdReader_.Open(fileName_);
        sdReader_.IsWavFile();
        return sdReader_.GetSize()/BUFF_SIZE_;
    }

    // ファイルの選択
    void SD_PlayerSkeleton::SelectFile()
    {
        selector_.DisplayFileList();
        title_.Draw("Select file");
        menu_.Inactivate(0);        // SELECT ボタン無効
        menu_.Inactivate(1);        // PLAY ボタン無効
        do
            if (selector_.Select(fileName_))
                menu_.Activate(1);  // PLAY ボタン有効
        while (!menu_.Touched(1));  // PLAY ボタンがタッチされるまで繰り返す

        selector_.Erase(0, 0, BG_LEFT_-4, 272);
        title_.Draw();
    }

    // パネルが操作された場合の処理
    void SD_PlayerSkeleton::DoIfHandled()
    {
        if (reset_ != NULL) reset_->DoIfTouched();
        Modefy();   // アプリ固有のパラメータ変更（仮想関数）
    }

    // メニューのいずれかのボタンがタッチされるまで DoIfHandled() を実行しながら待つ
    int SD_PlayerSkeleton::WaitTouched()
    {
        int touchNum;
        while (!menu_.GetTouchedNumber(touchNum)) DoIfHandled();
        return touchNum;
    }
}
