//---------------------------------------------------------------------
//  テスト用 FM 変調信号発生
//      FM 変調器１の搬送波： 70 kHz，変調信号 440 Hz, 振幅変動： 1 Hz
//      FM 変調器２の搬送波：130 kHz，変調信号：A1 から入力した信号
//      出力
//          A2 ：FM 変調波
//          D13：FM 変調器１の変調信号（モニタ用）
//
//  2021/03/03, Copyright (c) 2021 MIKAMI, Naoki
//---------------------------------------------------------------------

#include "SinOscIir.hpp"        // sin 発生器
#include "FM_Modulator.hpp"     // FM 変調器
#include "F446_ADC_Intr.hpp"    // ADC, 割込み利用
#include "F446_DAC_Dual.hpp"    // DAC を２つのチャンネルで使う場合
#include "IirDcCut.hpp"         // 直流成分除去用フィルタ
using namespace Mikami;
#pragma diag_suppress 870   // マルチバイト文字使用の警告抑制のため

const float AMP_ = 0.3f;    // 搬送波の振幅
const float FS_ = 500.0f;   // 標本化周波数，単位： kHz
const float T0_ = 1000/FS_; // 標本化間隔，単位：μs

AdcF446_Intr adc_(FS_, A1); // FM 変調器２の変調信号の入力端子：A1
DacDual dac_;               // 出力端子：A2, D13（固定）

SinOscIir modSignal_(440, T0_); // FM 変調器１の変調信号： 440 Hz の正弦波
// SinOscIir は float 型の演算精度の関係であまり低い周波数の正弦波は
// 発生できないため 100 Hz の正弦波を発生している
SinOscIir amplitude_(100, T0_); // FM 変調器１の変調信号の振幅を変化させるため

FmModulator modulator1_( 70.0e3f, T0_); // FM 変調器１
FmModulator modulator2_(130.0e3f, T0_); // FM 変調器２

// 入力の直流分除去用フィルタ
// 高域通過フィルタ
// バタワース特性
// 次数　　　　： 1 次
// 標本化周波数：500.00 kHz
// 遮断周波数　：  0.10 kHz
IirDcCut dcCut_(9.987442E-01f, 9.993721E-01f);  // 入力の直流成分除去用フィルタ

// AD 変換終了割り込みに対する割り込みサービス･ルーチン
void AdcIsr()
{
    static int16_t count = 0;
    static float amp1 = 0;

    float xn = adc_.Read();         // 外部の変調信号を入力
    // 100 回に 1 回正弦波を発生することにより，1 Hz の正弦波を発生している
    if (++count >= 100)
    {
        amp1 = 0.01f*(amplitude_.Generate() + 1.0f);
        count = 0;
    }
    // 変調信号の準備
    float mod1 = amp1*modSignal_.Generate(); // 振幅が 1 Hz で変動する正弦波
    float mod2 = 0.2f*dcCut_.Execute(xn);    // ADC に入力した信号から直流分除去

    // FM 変調された信号を生成
    float yn = AMP_*(modulator1_.Execute(mod1) + modulator2_.Execute(mod2));
    dac_.Write(yn, mod1*30);     // FM 変調された信号の出力
}

int main()
{
    printf("\r\nDAC (A2) に FM 変調された信号を出力します．\r\n");
    adc_.SetIntrVec(&AdcIsr);
    while (true) {}
}