//--------------------------------------------------------
//  STM32F446 内蔵の DAC 用のクラス
//      A2  (PA_4)： 左
//      D13 (PA_5)： 右
//  D13 は LD2 (緑) が接続されているため，高い電圧を出力した場合に
//  クリップされる．これを防止するためには SB21 を OFF にする
//
//  2020/08/11, Copyright (c) 2020 MIKAMI, Naoki
//--------------------------------------------------------

#include "mbed.h"

#ifndef STM32F446xx
#error Select NUCLEO-F446RE.
#endif

#ifndef F446_DAC_DUAL_HPP
#define F446_DAC_DUAL_HPP

namespace Mikami
{
    class DacDual
    {
    public:
        // コンストラクタ
        DacDual() : da1_(PA_4), da2_(PA_5)
        {
            DAC->CR = DAC_CR_EN1 | DAC_CR_TEN1 | DAC_CR_TSEL1 |
                      DAC_CR_EN2 | DAC_CR_TEN2 | DAC_CR_TSEL2;
        }

        virtual ~DacDual() {}

        // -1.0f <= data1, data2 <= 1.0f
        //      data1: 左, data2: 右
       void Write(float data1, float data2)
        {   WriteDac(ToUint16(data1), ToUint16(data2)); }

        // 0 <= data1, data2 <= 4095
        //      data1: 左, data2: 右
        void Write(uint16_t data1, uint16_t data2)
        {   WriteDac(__USAT(data1, BIT_WIDTH_),
                     __USAT(data2, BIT_WIDTH_)); }

    private:
        static const uint32_t SWTRIG_ = DAC_SWTRIGR_SWTRIG1
                                      | DAC_SWTRIGR_SWTRIG2;
        static const int BIT_WIDTH_ = 12;
        AnalogOut da1_, da2_;

        // 両方のチャンネルへ出力する
        void WriteDac(uint16_t val1, uint16_t val2)
        {
            DAC->DHR12RD = __PKHBT(val1, val2, 16);
            DAC->SWTRIGR = SWTRIG_;
        }

        // 飽和処理を行い uint16_t 型のデータを戻り値とする
        uint16_t ToUint16(float val)
        {
            return __USAT((val + 1.0f)*2047.0f, BIT_WIDTH_);
        }

        // コピー･コンストラクタ，代入演算子の禁止のため
        DacDual(const DacDual&);
        DacDual& operator=(const DacDual&);     
    };
}
#endif  // F446_DAC_DUAL_HPP