//-------------------------------------------------------------
//  F446 内蔵 用抽象基底クラス
//      ADC2 を使用（固定）
//
//  2020/07/24, Copyright (c) 2020 MIKAMI, Naoki
//-------------------------------------------------------------

#include "F446_ADC_Base.hpp"
#include "PeripheralPins.h" // PinMap_ADC を使う場合に必要
#pragma diag_suppress 870   // マルチバイト文字使用の警告抑制のため
// PeripheralPins.c は以下よりたどって行けば取得可能
// https://gitlab.exmachina.fr/fw-libs/mbed-os/tree/5.8.1

namespace Mikami
{
    // コンストラクタ
    AdcF446_Base::AdcF446_Base(float fSampling, PinName pin) : myAdc_(ADC2)
    {
        // pin に対応する GPIOx_MODER をアナログ･モードに設定する
        pin_function(pin, STM_MODE_ANALOG);
        // pin_function() が定義されている pinmap.c が含まれるディレクトリ：
        // mbed-dev\targets\TARGET_STM

        __HAL_RCC_ADC2_CLK_ENABLE();    // ADC2 にクロックを供給する
        // __HAL_RCC_ADC2_CLK_ENABLE() の定義：stm32f4xx_hal_rcc_ex.h

        // 1 チャンネルのみ使用の設定
        myAdc_->SQR1 &= ~ADC_SQR1_L;

        // pin に対応するチャンネルを使うための設定
        myAdc_->SQR3 = STM_PIN_CHANNEL(pinmap_function(pin, PinMap_ADC));
        // pinmap_function() のヘッダファイル： mbed\hal\pinmap.h
        // STM_PIN_CHANNEL() の定義：PinNamesTypes.h

        // ADC の CR1 の設定
        myAdc_->CR1 = 0x0;      // AD 変換終了割り込みを禁止
        // ADC の CR2 の設定
        myAdc_->CR2 = ADC_EXTERNALTRIGCONVEDGE_RISING   // 外部トリガの立ち上がりで開始される
                    | ADC_EXTERNALTRIGCONV_T8_TRGO      // 外部トリガ： Timer8 TRGO event
                    | ADC_CR2_ADON;                     // ADC を有効にする

        // AD 変換器の外部トリガに使うタイマ (TIM8) の設定
        SetTim8(fSampling);
    }

    // AD 変換器の外部トリガに使うタイマ (TIM8) の設定
    //      fSampling  標本化周波数 [kHz]
    void AdcF446_Base::SetTim8(float fSampling)
    {
        __HAL_RCC_TIM8_CLK_ENABLE();    // クロック供給. "stm32f4xx_hal_rcc.h" 参照
        TIM_TypeDef* const TIM = TIM8;

        TIM->CR2 = TIM_TRGO_UPDATE;     // Update event を TRGO (trigger output) とする

        float arrF = (SystemCoreClock/fSampling)/1000.0f;
        if (arrF >65535)
        {
            fprintf(stderr, "%8.2f kHz : 標本化周波数が低すぎます．\r\n", fSampling);
            while (true) {}
        }
        TIM->ARR = floor(arrF + 0.5f) - 1;  // Auto-reload レジスタの設定
        TIM->PSC = 0;                   // Prescaler の設定
        TIM->CR1 = TIM_CR1_CEN;         // TIM8 を有効にする
    }
}