//----------------------------------------------------------------------------
// SDR で使う FIR フィルタ，係数：対称
//      ● 処理の高速化のため，入力信号のバッファのサイズを 256，データを指す
//        インデックスを uint8_t 型としてリング･バッファを実現している．
//      ● さらに，係数が対称であることを利用して，先に x[k] + x[ORDER-k] と
//        いう加算を行ってから，係数の乗算を行うようにした．
//
// 2020/07/28, Copyright (c) 2020 MIKAMI, Naoki
//----------------------------------------------------------------------------

#include "mbed.h"

#ifndef FIR_DIRECT_FAST_SYMMETRY_HPP
#define FIR_DIRECT_FAST_SYMMETRY_HPP
namespace Mikami
{
    class FirFastSymmetry
    {
    public:
        // コンストラクタ
        //      order   フィルタの次数，次数は偶数とする
        //      hk[]    フィルタの係数
        FirFastSymmetry(uint8_t order, const float hk[])
            : ORDER_(order), ORDER2_(order/2), hm_(hk), index_(255)
        {   for (int k=0; k<SIZE_; k++) un_[k] = 0; }

        // 入力信号をバッファへ書き込む
        void Store(float xIn) { un_[++index_] = xIn; }

        // FIR フィルタの実行
        float Execute()
        {
            __IO uint8_t ptrM = index_;
            __IO uint8_t ptrP = index_ - ORDER_;
            float acc = 0;
            for (int k=0; k<ORDER2_; k++)
                acc += hm_[k]*(un_[ptrM--] + un_[ptrP++]);
            acc += hm_[ORDER2_]*un_[ptrM];
            return acc;
        }

    private:
        static const int SIZE_ = 256;   // 入力信号のバッファのサイズ
        const uint8_t ORDER_;           // フィルタの次数
        const uint8_t ORDER2_;          // フィルタの次数の半分
        const float *const hm_;         // フィルタの係数
        float un_[SIZE_];               // 入力信号のバッファ
        __IO uint8_t index_;            // 最新の入力信号を示すインデックス

        // コピー･コンストラクタ，代入演算子の禁止のため
        FirFastSymmetry(const FirFastSymmetry&);
        FirFastSymmetry& operator=(const FirFastSymmetry&);
    };
}
#endif  // FIR_DIRECT_FAST_SYMMETRY_HPP