//--------------------------------------------------------------
// Reverb unit
// Copyright (c) 2014 MIKAMI, Naoki,  2014/07/28
//--------------------------------------------------------------

#ifndef REVERB_UNIT_HPP
#define REVERB_UNIT_HPP

#include "mbed.h"

namespace Mikami
{
    // Base class for reverb unit
    template<int delay> class Reverb
    {
    private:
        int ptr_;
        float un_[delay];    // for delay

        Reverb(const Reverb&);
        Reverb& operator=(const Reverb&);
    protected:
        float Get() { return un_[ptr_]; }
        void Set(float x)
        {
            un_[ptr_] = x;   
            if (++ptr_ >=  delay) ptr_ = 0;
        }
    public:
        // Constructor
        Reverb() : ptr_(0) { Clear(); }
        
        // Execute of filter (pure virtual function)
        virtual float Execute(float x) = 0;

        // Clear internal delay elements
        void Clear()
        { for (int n=0; n<delay; n++) un_[n] = 0; }
    };

    // Reverb unit using comb filter
    template<int delay> class CombFilter : public Reverb<delay>
    {
    private:
        const float G0_;
    public:
        // Constructor
        CombFilter(float g) : G0_(g) {}

        // Execute comb filter 
        virtual float Execute(float x)
        {
            float yn = Reverb<delay>::Get();
            Reverb<delay>::Set(x + G0_*yn);
            return yn;
         }
    };

    // Reverb unit using allpass filter
    template<int delay> class AllPassFilter : public Reverb<delay>
    {
    private:
        const float G0_;
    public:
        // Constructor
        AllPassFilter(float g) : G0_(g) {}

        // Execute allpass filter 
        virtual float Execute(float x)
        {
            float un = x + G0_*Reverb<delay>::Get();
            float yn = -G0_*un + Reverb<delay>::Get();
            Reverb<delay>::Set(un);
            return yn;
        }
    };
}
#endif  // REVERB_UNIT_HPP
