//----------------------------------------------------------------------
//  Ticker と同じような機能のほかに，割込みの時間間隔を設定するメンバ関数を
//  追加した行うクラス
//
//  2020/02/22, Copyright (c) 2020 MIKAMI, Naoki
//----------------------------------------------------------------------

#include "MyTicker14.hpp"
using namespace Mikami;

MyTicker14::MyTicker14()
{
    MBED_ASSERT(!created_);  // このオブジェクトが複数生成されないようにするため

    // TIM14 にクロック供給．"stm32f4xx_hal_rcc_ex.h" 参照
    __HAL_RCC_TIM14_CLK_ENABLE();

    // TIM14 を駆動するクロックの周波数を求める
    if ((RCC->DCKCFGR & RCC_DCKCFGR_TIMPRE_Msk) == RCC_DCKCFGR_TIMPRE)
        intClock_ = SystemCoreClock;
    else
        intClock_ = SystemCoreClock/2;
}

// 割込みサービスルーチンの割当てと割込み周期（μs 単位）の設定
void MyTicker14::Attach_us(void (*Func)(), uint16_t time)
{
    SetPeriod_us(time);
    SetCommon(Func);
}

// 割込みサービスルーチンの割り当てと割込み周期（ms 単位）の設定
void MyTicker14::Attach_ms(void (*Func)(), uint16_t time)
{
    SetPeriod_ms(time);
    SetCommon(Func);
}

// 割込み周期（μs 単位）の設定
void MyTicker14::SetPeriod_us(uint16_t time)
{
    tim_->ARR = time - 1;               // Auto-reload
    tim_->PSC = intClock_/1000000 - 1;  // Prescaler
}

// 割込み周期（ms 単位）の設定
void MyTicker14::SetPeriod_ms(uint16_t time)
{
    tim_->ARR = 10*time - 1;            // Auto-reload
    tim_->PSC = intClock_/10000 - 1;    // Prescaler
}

// 割込みサービス･ルーチン
void MyTicker14::Isr()
{
    // TIM14 の Update 割込みでない場合は return
    if ((tim_->SR & TIM_SR_UIF_Msk) != TIM_SR_UIF) return;

    tim_->SR &= ~TIM_SR_UIF;   // Update 割込みをクリア
    fp_();  // Attach_us(), Attach_ms() の引数として渡された関数を実行
}

// タイマ設定の共通の処理
void MyTicker14::SetCommon(void (*Func)())
{
    // CR1 の ARPE を 1 にすることで，ARR の更新を Update event (UVE)
    // に合わせて行うように設定できる
    // リファレンス･マニュアル (RM0390) p.587 参照
    tim_->CR1 = TIM_CR1_CEN             // Counter 有効
              | TIM_CR1_ARPE;           // ARR のバッファ有効
    // ARR のバッファが無効（ARPE = 0）だと，ARR を再設定してタイマ割込み周期を
    // 変える場合，TIM14 のカウンタが 0xFFFF までカウントしてしまう場合が出てくる

    fp_ = Func; // Attach_us(), Attach_ms() の引数として渡された関数を割り当てる

    // メンバ関数を割込みサービス･ルーチンとして NVIC_SetVector() に渡す場合，
    // そのメンバ関数は static でなければならない
    NVIC_SetVector(TIM8_TRG_COM_TIM14_IRQn,
                   (uint32_t)Isr);              // "core_cm4.h" 参照
    NVIC_EnableIRQ(TIM8_TRG_COM_TIM14_IRQn);    // "core_cm4.h" 参照

    Enable();       // Update 割込み有効
}

// static メンバの実体
void (*MyTicker14::fp_)();
TIM_TypeDef* const MyTicker14::tim_ = TIM14;
bool MyTicker14::created_ = false;

