//----------------------------------------------------------------------
//  TIM13 の Update event (UEV) 割込みを使って，
//  Ticker と同じような機能を持つクラス
//
//  2020/01/25, Copyright (c) 2020 MIKAMI, Naoki
//----------------------------------------------------------------------

#include "MyTicker13.hpp"
using namespace Mikami;

MyTicker13::MyTicker13()
{
    MBED_ASSERT(!created_);  // このオブジェクトが複数生成されないようにするため

    // TIM13 にクロック供給．"stm32f4xx_hal_rcc_ex.h" 参照
    __HAL_RCC_TIM13_CLK_ENABLE();

    // TIM13 を駆動するクロックの周波数を求める
    if ((RCC->DCKCFGR & RCC_DCKCFGR_TIMPRE_Msk) == RCC_DCKCFGR_TIMPRE)
        intClock_ = SystemCoreClock;
    else
        intClock_ = SystemCoreClock/2;
    created_ = true;    // MyTicker13 のオブジェクトがすでに生成されていることを示す
}

// 割込みサービスルーチンの割当てと割込み周期（μs 単位）の設定
void MyTicker13::Attach_us(void (*Func)(), uint16_t time)
{
    tim_->ARR = time - 1;               // Auto-reload
    tim_->PSC = intClock_/1000000 - 1;  // Prescaler
    SetCommon(Func);
}

// 割込みサービスルーチンの割り当てと割込み周期（ms 単位）の設定
void MyTicker13::Attach_ms(void (*Func)(), uint16_t time)
{
    tim_->ARR = 10*time - 1;            // Auto-reload
    tim_->PSC = intClock_/10000 - 1;    // Prescaler
    SetCommon(Func);
}

// 割込みサービス･ルーチン
void MyTicker13::Isr()
{
    // TIM13 の Update 割込みでない場合は return
    if ((tim_->SR & TIM_SR_UIF_Msk) != TIM_SR_UIF) return;

    tim_->SR &= ~TIM_SR_UIF;   // Update 割込みをクリア
    fp();   // Attach_us(), Attach_ms() の引数として渡された関数を実行
}

// タイマ設定の共通の処理
void MyTicker13::SetCommon(void (*Func)())
{
    tim_->CR1 = TIM_CR1_CEN;    // Counter 有効

    fp = Func;  // Attach_us(), Attach_ms() の引数として渡された関数を割り当てる

    NVIC_SetVector(TIM8_UP_TIM13_IRQn, (uint32_t)Isr);  // "core_cm4.h" 参照
    NVIC_EnableIRQ(TIM8_UP_TIM13_IRQn);                 // "core_cm4.h" 参照

    Enable();        // Update 割込み有効
}

// static メンバの実体
void (*MyTicker13::fp)();
TIM_TypeDef* const MyTicker13::tim_ = TIM13;
bool MyTicker13::created_ = false;

