//------------------------------------------------------------------------
//  ディジタルフィルタの周波数特性を，周波数軸が対数スケールで描画するクラス（ヘッダ）
//  FrqRespDrawer class (header)
//
//  2017/03/16, Copyright (c) 2017 MIKAMI, Naoki
//------------------------------------------------------------------------

#ifndef F746_FRQ_RESP_DRAWER_HPP
#define F746_FRQ_RESP_DRAWER_HPP

#include "NumericLabel.hpp"
#include "FrequancyResponseBase.hpp"

namespace Mikami
{
    class FrqRespDrawer
    {
    public:
        // 横軸の目盛値を描画する際に使う構造体
        struct AxisX_Char { uint16_t frq; string str; };
        
        // Constructor
        FrqRespDrawer(uint16_t orgX, float min, float max,
                      uint16_t dec,     // 周波数の 10 倍に対応する長さ（ピクセル数）
                      uint16_t orgY,
                      float minDb,      // 表示する dB 値の最小値
                      float maxDb,      // 表示する dB 値の最大値
                      float db1Pixel,   // 1 dB に対応するピクセル数
                      float ySpace,     // 目盛線の縦方向の間隔（dB 単位）
                      float fs,
                      uint32_t lineColor = 0xFF00B0FF,
                      uint32_t axisColor = LCD_COLOR_LIGHTGRAY,
                      uint32_t backColor = GuiBase::ENUM_BACK)
            : lcd_(GuiBase::GetLcd()),
              ORGX_(orgX), MIN_(min), MAX_(max), DEC_(dec),
              ORGY_(orgY), MIN_DB_(minDb), MAX_DB_(maxDb),
              DB1_(db1Pixel), Y_SPACE_(db1Pixel*ySpace), FS_(fs),
              LINE_COLOR_(lineColor),
              AXIS_COLOR_(axisColor),
              BACK_COLOR_(backColor) {}

        // 周波数に対応する x 座標値の取得
        int X(float frq)
        {   return Round(ORGX_ + DEC_*log10f(frq/MIN_));  }

        // x 座標値を周波数に変換
        float PosToFrq(uint16_t x)
        {   return MIN_*powf(10.0f, (x - ORGX_)/(float)DEC_); }

        // 目盛線の描画
        void DrawAxis();

        // 横軸の目盛値の表示
        void DrawNumericX(AxisX_Char xChar[], int nDisp, int offsetY,
                          string str, sFONT &fonts = Font12,
                          uint32_t textColor = LCD_COLOR_WHITE);

        // 縦軸の目盛値の表示
        void DrawNumericY(int offsetX, int offsetY, uint16_t d_dB,
                          const char fmt[], sFONT &fonts = Font12,
                          uint32_t textColor = LCD_COLOR_WHITE,
                          string str = "[dB]");

        // 周波数特性のグラフの描画
        void DrawGraph(FrequencyResponse &frqResp, uint32_t color);
        void DrawGraph(FrequencyResponse &frqResp)
        {   DrawGraph(frqResp, LINE_COLOR_);}

        // 消去
        //      upDb : 消去する範囲の上限（dB 単位）
        void Erase(float upDb = 0);

    private:
        LCD_DISCO_F746NG &lcd_;
        const uint16_t ORGX_;   // 横軸の目盛の最小値に対応する位置
        const float MIN_;       // 横軸の目盛の最小値
        const float MAX_;       // 横軸の目盛の最大値
        const uint16_t DEC_;    // 周波数の 10 倍に対応する長さ (pixels)
        const uint16_t ORGY_;   // 縦軸の目盛の最小値に対応する位置
        const float MIN_DB_;    // 縦軸の目盛の最小値 [dB]
        const float MAX_DB_;    // 縦軸の目盛の最大値 [dB]
        const float DB1_;       // 1 dB に対応する pixels 数
        const float Y_SPACE_;   // 縦軸の目盛線の間隔に対応する pixels 数
        const float FS_;        // 標本化周波数
        const uint32_t LINE_COLOR_;
        const uint32_t AXIS_COLOR_;
        const uint32_t BACK_COLOR_;
        
        // 丸め
        int Round(float x) { return x + 0.5f - (x < 0); }  

        // 10 のべき乗かどうかの検査
        bool PowersOf10(float  x)
        {   return fabsf(log10f(x) - Round(log10f(x))) < 0.01f; }

        // disallow copy constructor and assignment operator
        FrqRespDrawer(const FrqRespDrawer&);
        FrqRespDrawer& operator=(const FrqRespDrawer&);
    };
}
#endif  // F746_FRQ_RESP_DRAWER_HPP
