//-------------------------------------------------------
// Class for display spectrum
// Copyright (c) 2015 MIKAMI, Naoki,  2015/10/27
//-------------------------------------------------------

#include "SpectrumDisplay.hpp"

namespace Mikami
{
    SpectrumDisplay::SpectrumDisplay(
            LCD_DISCO_F746NG &lcd,
            int nFft, int x0, int y0,
            float db1, int bin, float maxDb, int fs)
        : N_FFT_(nFft), X0_(x0), Y0_(y0),
          DB1_(db1), BIN_(bin), MAX_DB_(maxDb), FS_(fs),
          LCD_(&lcd)
    {
        AxisX();
        AxisY();
    }

    void SpectrumDisplay::BarChart(float db[], uint32_t backColor)
    {
        const float OFFSET = 60.0f;
        for (int n=1; n<=N_FFT_/2; n++)
        {
            float h = ((db[n] + OFFSET) >= 0)? db[n] + OFFSET : 0;
            
            int y = Y0_ - (int)(h*DB1_);
            LCD_->SetTextColor(LCD_COLOR_CYAN);
            LCD_->DrawLine(X0_+n, Y0_-1, X0_+n, y);
            LCD_->SetTextColor(backColor);
            LCD_->DrawLine(X0_+n, y-1, X0_+n, Y0_-(int)(MAX_DB_*DB1_));
        }
        LCD_->SetTextColor(LCD_COLOR_YELLOW);
        LCD_->DrawLine(X0_, Y0_, X0_+BIN_*N_FFT_/2, Y0_);
    }

    void SpectrumDisplay::LineChart(float db[], uint32_t backColor)
    {
        const float OFFSET = 60.0f;
        LCD_->SetTextColor(backColor);
        LCD_->FillRect(X0_+1, Y0_-(int)(MAX_DB_*DB1_),
                       N_FFT_*BIN_/2, MAX_DB_*DB1_);

        float h = ((db[1] + OFFSET) > 0)? db[1] + OFFSET : 0;
        int y1 = Y0_ - (int)(h*DB1_);
        for (int n=1; n<N_FFT_/2; n++)
        {
            float h2 = ((db[n+1] + OFFSET) > 0)? db[n+1] + OFFSET : 0;
            if (h2 > MAX_DB_) h2 = MAX_DB_;
            int y2 = Y0_ - (int)(h2*DB1_);
            LCD_->SetTextColor(LCD_COLOR_CYAN);
            LCD_->DrawLine(X0_+n, y1, X0_+n+1, y2);
            y1 = y2;
        }
        LCD_->SetTextColor(LCD_COLOR_YELLOW);
        LCD_->DrawLine(X0_, Y0_, X0_+BIN_*N_FFT_/2, Y0_);
    }

    // Clear Spectrum
    void SpectrumDisplay::Clear(uint32_t backColor)
    {
        LCD_->SetTextColor(backColor);
        LCD_->FillRect(X0_+1, Y0_-(int)(MAX_DB_*DB1_),
                       N_FFT_*BIN_/2, MAX_DB_*DB1_);
    }

    // x-axis
    void SpectrumDisplay::AxisX()
    {
        LCD_->SetFont(&Font12);

        LCD_->SetTextColor(LCD_COLOR_YELLOW);
        LCD_->DrawLine(X0_, Y0_, X0_+BIN_*N_FFT_/2, Y0_);
        float dx = BIN_*(N_FFT_*1000.0f)/(float)FS_;
        for (int n=0; n<=(FS_/1000)/2; n++)
        {
            int xTick = X0_ + (int)(dx*n + 0.5f);
            char s[5];
            sprintf(s, "%d", n);
            LCD_->DisplayStringAt(xTick-3, Y0_+10, (uint8_t *)s, LEFT_MODE);
            LCD_->DrawLine(xTick, Y0_, xTick, Y0_+5);
        }
        LCD_->DisplayStringAt(X0_+74, Y0_+24, (uint8_t *)"Frequency [kHz]", LEFT_MODE);
    }

    // y-axis
    void SpectrumDisplay::AxisY()
    {
        LCD_->SetFont(&Font12);
        
        LCD_->SetTextColor(LCD_COLOR_YELLOW);
        LCD_->DrawLine(X0_, Y0_+5, X0_, Y0_-(int)(MAX_DB_*DB1_));
        for (int n=0; n<=(int)MAX_DB_; n+=20)
        {
            int yTick = Y0_-(int)(DB1_*n);
            char s[5];
            sprintf(s, "%3d", n-60);
            LCD_->DisplayStringAt(X0_-30, yTick-5, (uint8_t *)s, LEFT_MODE);
            LCD_->DrawLine(X0_-5, yTick, X0_, yTick);
        }
        LCD_->DisplayStringAt(X0_-27, Y0_-(int)(DB1_*MAX_DB_)-18, (uint8_t *)"[dB]", LEFT_MODE);
    }
}

