//-----------------------------------------------------------
//  ディジタルフィルタ実験のためのプラットフォームの
//  LCD パネル表示と入力信号の周波数設定で使う
//
//  2018/11/03, Copyright (c) 2017 MIKAMI, Naoki
//-----------------------------------------------------------

#ifndef PANEL_MANIPULATOR_HPP
#define PANEL_MANIPULATOR_HPP

#include "F746_GUI.hpp"
#include "NumericUpDown.hpp"
#include "Zoom.hpp"
#include "WaveformDisplay.hpp"
#include "Trigger.hpp"

namespace Mikami
{
    class LcdPanel
    {
    public:
        //  ts  : 標本化間隔（単位：μs）
        LcdPanel(float ts)
            : TS_(ts), lcd_(GuiBase::GetLcd()),
              xnD_(N_DISP_BUF_), frq_(1000),
              dPhi_(PI2_*frq_*TS_), phi_(0), factor_(1),
              noiseLevel_(0)
        {
            Label title(240, 10, "Sin+Noise Generator", Label::CENTER, Font16);

            // 表示するデータ数： 400個
            waveDisp_ = new WaveformDisplay(lcd_, 30, 90, xnD_, 400, 8,
                                            AXIS_COLOR_,
                                            LINE_COLOR1_, LINE_COLOR2_,
                                            BACK_COLOR_);

            // GUI 設定
            ud_ = new NumericUpDown(4, 0, 271, frq_, 9999, 100, "Hz");
            bar_ = new SeekBar(230, 250, 230,
                               log10f(MIN_FRQ_), log10f(MAX_FRQ_), log10f(1000),
                               "0.1", "1.0", "10.0");
            // ノイズ･レベルの設定用
            nLevel_ = new SeekBar(270, 185, 190, 0, 1.0f, 0, "0", "0.5", "1.0");

            // 時間軸の拡大率は 8 倍まで
            zmButton_ = new ZoomButton(180, 170, 30, 30, 8);
        }

        // 正弦波の発生で使う値の更新し，取得する
        float GetPhi()
        {
            if ((phi_ += dPhi_) > PI2_) phi_ -= PI2_;
            return phi_;
        }

        // 周波数設定の更新とズームイン／アウト･ボタンの処理
        void Update()
        {        
            if (bar_->Slide())  // 周波数設定用 SeekBar をスライドした場合の処理
            {
                frq_ = (int)(powf(10, bar_->GetValue())+0.5f);
                if (frq_ >= MAX_FRQ_) frq_ = MAX_FRQ_ - 1;
                ud_->Set(frq_); // SeekBar の数値を NumericUpDown に表示する
                dPhi_ = PI2_*frq_*TS_;
            }

            if (nLevel_->Slide())  // ノイズ･レベル設定用 SeekBar をスライドした場合の処理
                noiseLevel_ = nLevel_->GetValue();

            if (ud_->Touched())     // NumericUpDown をタッチした場合の処理
            {
                frq_ = ud_->Get();
                bar_->Draw(log10f(frq_));  // NumericUpDown の数値で SeekBar のツマミの位置を設定
                dPhi_ = PI2_*frq_*TS_;
            }

            zmButton_->Touched(factor_);   // ズームインとズームアウト
        }

        // 入出力信号のデータを表示用バッファへ格納
        void Store(int16_t xn, int n)
        {
            // N_DISP_BUF_ (= 1000) 個を越えたデータは格納しない
            if (n >= N_DISP_BUF_) return;

            xnD_[n] = xn;
        }
        
        // 入出力信号の波形の表示（トリガ条件を満足したところから表示する）
        void Display() { waveDisp_->Execute(Trigger(xnD_, 1), factor_); }

        // ノイズの大きさの取得
        float GetNoiseLevel() { return noiseLevel_; }

    private:
        static const float PI_ = 3.1415926536f;
        static const float PI2_ = 2.0f*PI_;
        static const int MIN_FRQ_ = 100;        // 発生する周波数の最小値
        static const int MAX_FRQ_ = 10000;      // 発生する周波数の最大値
        static const uint32_t BACK_COLOR_ = GuiBase::ENUM_BACK;
        static const uint32_t AXIS_COLOR_ = 0xFFCCFFFF;
        static const uint32_t LINE_COLOR1_ = LCD_COLOR_CYAN;
        static const uint32_t LINE_COLOR2_ = LCD_COLOR_MAGENTA;
        static const int N_DISP_BUF_ = 1000;    // 波形表示用のバッファのサイズ

        const float TS_;    // 標本化間隔

        LCD_DISCO_F746NG &lcd_;
        WaveformDisplay *waveDisp_;
        NumericUpDown *ud_;     // 入力信号の周波数設定用（1 Hz 単位での設定用）
        SeekBar *bar_;          // 入力信号の周波数設定用（概略の設定用）
        SeekBar *nLevel_;       // ノイズの大きさの設定用
        ZoomButton *zmButton_;  // 波形表示：ズームイン/アウト用

        Array<int16_t> xnD_;    // 波形表示で使うバッファ（入力信号用）
        int frq_;               // 現在の入力信号の周波数，初期値は 1 kHz
        float dPhi_, phi_;      // 入力信号の生成で使う変数
        int factor_;            // 波形表示の際の標本化間隔に対応するピクセル数
        float noiseLevel_;      // ノイズの大きさ
    };
}
#endif  // PANEL_MANIPULATOR_HPP
