//--------------------------------------------------------------
//  FileSelector class
//      SD カード内のファイル名の一覧を表示し，ファイルを選択する
//
//  2016/06/30, Copyright (c) 2016 MIKAMI, Naoki
//--------------------------------------------------------------

#include "FileSelectorWav.hpp"

namespace Mikami
{
    bool FileSelector::CreateTable()
    {
        DIR* dp = opendir("/sd");
        fileCount_ = 0; 
        if (dp != NULL)
        {
            dirent* entry;
            for (int n=0; n<256; n++)
            {
                entry = readdir(dp);
                if (entry == NULL) break;

                string strName = entry->d_name;
                if ( (strName.find(".wav") != string::npos) ||
                     (strName.find(".WAV") != string::npos) )
                {
                    sdReader_.Open(strName);        // ファイルオープン

                    // PCM，16 ビットステレオ，標本化周波数 44.1 kHz 以外のファイルは除外
                    if (sdReader_.IsWavFile())
                        fileNames_[fileCount_++] = strName;
                    sdReader_.Close();
                }

                if (fileCount_ >= MAX_FILES_) break;
            }
            closedir(dp); 
        }
        else
            return false;

        if (fileCount_ == 0) return false;

        if (rect_ != NULL) delete rect_;
        Array<string> nonString(fileCount_, "");
        rect_ = new ButtonGroup(X_, Y_, W_H_, W_H_, fileCount_,
                                nonString, 0, V_L_-W_H_, 1,
                                -1, Font12, 0, GuiBase::ENUM_BACK,
                                BASE_COLOR_, TOUCHED_COLOR_);
        for (int n=0; n<fileCount_; n++) rect_->Erase(n);
        CreateLabels();   
        prevFileCount_ = fileCount_;
        return true;
    }

    // ファイルを選択する
    bool FileSelector::Select(string &fileName)
    {
        int n;
        if (rect_->GetTouchedNumber(n))
        {
            fileNameLabels_[n]->Draw(GetFileNameNoExt(n), TOUCHED_COLOR_);
            if ((prev_ >= 0) && (prev_ != n))
                fileNameLabels_[prev_]->Draw(GetFileNameNoExt(prev_));
            prev_ = n;
            fileName = fileNames_[n];
            return true;
        }
        else
            return false;
    }

    // ファイルの一覧の表示
    void FileSelector::DisplayFileList(bool sortEnable)
    {
        if (sortEnable)
            std::sort((string *)fileNames_,
                      (string *)fileNames_+fileCount_); 
        
        Erase(X_, Y_, MAX_NAME_LENGTH_*((sFONT *)(&Font16))->Width, 272-Y_);
        rect_->DrawAll();
        for (int n=0; n<fileCount_; n++)
            fileNameLabels_[n]->Draw(GetFileNameNoExt(n));
    }

    // ファイルの一覧の消去
    void FileSelector::Erase(uint16_t x, uint16_t y, uint16_t width, uint16_t height,
               uint32_t color)
    {
        lcd_->SetTextColor(color);
        lcd_->FillRect(x, y, width, height);
    }

    // Label を生成
    void FileSelector::CreateLabels()
    {
        fileNameLabels_.SetSize(fileCount_);
                    
        for (int n=0; n<fileCount_; n++)
            fileNameLabels_[n] = new Label(X_+30, Y_+5+V_L_*n, "",
                                           Label::LEFT, Font16,
                                           BASE_COLOR_);
    }

    // 拡張子を削除した文字列を取得
    string FileSelector::GetFileNameNoExt(int n)
    {
        string name = fileNames_[n];
        name.erase(name.find("."));
        return name.substr(0, MAX_NAME_LENGTH_);
    }

}
