//-----------------------------------------------------------
// SD_BitmapReader クラス（ヘッダ）
//      SD カードの *.bmp ファイルの内容を読み出す
//      ファイルの画素数：480×272 ピクセル
//      だだし，ファイル名が日本語のものが除く
//
//  2018/10/23, Copyright (c) 2018 MIKAMI, Naoki
//-----------------------------------------------------------

#ifndef SD_BITMAP_READER_HPP
#define SD_BITMAP_READER_HPP

#include "mbed.h"
#include "SDFileSystem.h"
#include "BlinkLabel.hpp"
#include "Array.hpp"
#include <string>

namespace Mikami
{
    class SD_BitmapReader
    {
    public:
        SD_BitmapReader(uint16_t maxWidth, uint16_t maxHeight)
            : STR_("sd"), sd_(new SDFileSystem(STR_.c_str())),
              maxWidth_(maxWidth), maxHeight_(maxHeight)
        { sd_->mount(); }

        virtual ~SD_BitmapReader();

        bool Open(const string fileName);
        
        void Close() { fclose(fp_); }

        // 条件に合うビットマップ･ファイルかどうか調べる
        bool IsAllowed();

        // ファイルの読み出し
        //      size    読み出すデータのバイト数
        size_t ReadFile(uint8_t data[], size_t size)
        { return fread(data, 1, size, fp_); }
        
        // 画像の高さの取得
        uint32_t  GetHeight(uint8_t *bmp)
        { return *(uint32_t *)(bmp+22); }

        // 画像の幅の取得
        uint32_t  GetWidth(uint8_t *bmp)
        { return *(uint32_t *)(bmp+18); }

        // 画像のサイズの取得
        uint32_t  GetSize(uint8_t *bmp)
        { return *(uint32_t *)(bmp+34); }

        // 画像の高さの設定
        void SetHeight(uint8_t *bmp, uint32_t height)
        { *(uint32_t *)(bmp+22) = height; }

        // 画像の幅の設定
        void SetWidth(uint8_t *bmp, uint32_t width)
        { *(uint32_t *)(bmp+18) = width; }

        // 画像のサイズの設定
        void SetSize(uint8_t *bmp, uint32_t size)
        { *(uint32_t *)(bmp+34) = size; }

    private:
        const string STR_;

        SDFileSystem *sd_;
        FILE *fp_;
        
        uint16_t maxWidth_;
        uint16_t maxHeight_;

        void ErrorMsg(char msg[])
        {   BlinkLabel errLabel(240, 100, msg, Label::CENTER); }
    };
}
#endif  // SD_BITMAP_READER_HPP
