//--------------------------------------------------------------
//  MEMS マイクまたはラインから CODEC を使って入力した信号の波形の表示
//      標本化周波数： 16 kHz
//
//  使用しているライブラリのリビジョン：
//          F746_GUI            Rev.33
//          F746_SAI_IO         Rev.12
//          mbed                Rev.161
//
//  2018/03/12, Copyright (c) 2018 MIKAMI, Naoki
//--------------------------------------------------------------

#include "F746_GUI.hpp"
#include "SAI_InOut.hpp"
#include "WaveformDisplay.hpp"
#include "Trigger.hpp"

int main()
{
    const uint32_t BACK_COLOR = GuiBase::ENUM_BACK;
    const uint32_t AXIS_COLOR = 0xFFCCFFFF;
    const uint32_t LINE_COLOR = LCD_COLOR_CYAN;

    const uint32_t N_DATA_SAI = 600;        // SAI で使うデータ数
    const uint32_t N_DATA = N_DATA_SAI*2/3; // 表示するデータ数
    SaiIO mySai(SaiIO::INPUT, N_DATA_SAI, AUDIO_FREQUENCY_16K,
                INPUT_DEVICE_DIGITAL_MICROPHONE_2);

    LCD_DISCO_F746NG &lcd = GuiBase::GetLcd();
    lcd.Clear(BACK_COLOR);

    Label labelTitle(240, 10, "Simplified Oscilloscope",
                     Label::CENTER, Font16);
    Label labelTrigg(20, 250);

    const uint16_t B_LEFT = 240;
    const uint16_t B_TOP = 235;
    const uint16_t B_WIDTH = 55;
    const uint16_t B_HEIGHT = 35;

    ButtonGroup inputSw(B_LEFT, B_TOP, B_WIDTH, B_HEIGHT,
                        2, (string[]){"MIC", "LINE"}, 0, 0, 2, 0);

    ButtonGroup runStop(B_LEFT+B_WIDTH*2+10, B_TOP, B_WIDTH, B_HEIGHT,
                        2, (string[]){"RUN", "STOP"}, 0, 0, 2, 0);

    const int X_WAV = 30;   // x 方向の原点
    const int Y_WAV = 100;  // y 方向の原点（上部に表示される波形）
    Array<int16_t> xnL(N_DATA), xnR(N_DATA);    // 表示で使うデータ
    WaveformDisplay waveDispL(lcd, X_WAV, Y_WAV, xnL, 7,
                              AXIS_COLOR, LINE_COLOR, BACK_COLOR);
    WaveformDisplay waveDispR(lcd, X_WAV, Y_WAV+80, xnR, 7,
                              AXIS_COLOR, LINE_COLOR, BACK_COLOR);
    Label labelL(445, Y_WAV- 8, "L", Label::LEFT, Font16);
    Label labelR(445, Y_WAV+72, "R", Label::LEFT, Font16);

    Array<int16_t> snL(N_DATA_SAI), snR(N_DATA_SAI);
    Trigger myTrigger(snL, snR, xnL, xnR, 64);
    int inSelect = 0;   // 初期値：入力デバイスをマイクにする

    mySai.RecordIn();   // 入力開始
    bool run = true;    // 入力実行中

    while (true)
    {
        int sw = 0;
        if (runStop.GetTouchedNumber(sw))
        {
            bool on = (sw == 0) ? true : false;
            if (on && !run) mySai.ResumeIn();   // 入力再開
            if (!on && run) mySai.PauseIn();    // 入力中断
            run = on;
        }

        int swNow;
        if (inputSw.GetTouchedNumber(swNow) && (swNow != inSelect))
        {
            mySai.SwitchInputDevice(swNow); // 入力デバイスの設定
            inSelect = swNow;
        }

        if (mySai.IsCaptured())     // 入力データがそろってから波形を表示する
        {
            for (int n=0; n<N_DATA_SAI; n++)
                mySai.Input(snL[n], snR[n]);

            bool trigg = myTrigger.Execute();   // トリガと同等の操作

            if (trigg) labelTrigg.Draw("Triggered");
            else       labelTrigg.Draw("Not Triggered");

            waveDispL.Execute();    // 左チャンネルの波形表示
            waveDispR.Execute();    // 右チャンネルの波形表示
        }    
    }
}
