//-----------------------------------------------------------
//  SaiIO class (Header)
//  2017/03/17, Copyright (c) 2017 MIKAMI, Naoki
//-----------------------------------------------------------

#ifndef F746_SAI_IO_HPP
#define F746_SAI_IO_HPP

#include "mbed.h"
#include "stm32746g_discovery_audio.h"
#include "BSP_AudioIn_Overwrite.hpp"
#include "BSP_AudioOut_Overwrite.hpp"
#include "Array.hpp"

namespace Mikami
{
    class SaiIO
    {
    public:
        enum InOutBoth { INPUT,     // input only
                         OUTPUT,    // output only
                         BOTH };    // input and output
        // Constructor
        //      inputDevice: INPUT_DEVICE_DIGITAL_MICROPHONE_2 or
        //                   INPUT_DEVICE_INPUT_LINE_1
        //      inputDevice == 0 : not use input device
        SaiIO(InOutBoth ioBoth, int size, int fs,
              uint16_t inputDevice = 0);
        virtual ~SaiIO() {}
 
        int32_t GetLength() { return nData_; }
        
        void RecordIn();
        // sw = 0: DIGITAL_MICROPHONE_2
        //      1: LINE_1
        void SwitchInputDevice(int sw);

        bool IsCaptured();
        // Input using SAI
        void Input(int16_t &xL, int16_t &xR)
        {   (this->*InputFp)(xL, xR); }

        void StopIn()   { BSP_AUDIO_IN_Stop(CODEC_PDWN_SW); }
        void PauseIn()  { BSP_AUDIO_IN_Pause(); }
        void ResumeIn() { BSP_AUDIO_IN_Resume(); }

        void PlayOut();
        bool IsXferred();
        // Output using SAI
        void Output(int16_t xL, int16_t xR);
      
        void StopOut()   { BSP_AUDIO_OUT_Stop(CODEC_PDWN_SW); }
        void PauseOut()  { BSP_AUDIO_OUT_Pause(); }
        void ResumeOut() { BSP_AUDIO_OUT_Resume(); }
       
        // IF you use both input and output of SAI,
        // you can use following function
        bool IsCompleted()
        {   return IsCaptured() && IsXferred(); }

        // Following two member functions are called from
        // callback functions in "BSP_AudioIn_Overwrite.cpp"
        static void Captured1st() { Captured(0); }
        static void Captured2nd() { Captured(bufferSize_/2); }

        // Following two member functions are called from
        // callback functions in "BSP_AudioOut_Overwrite.cpp"
        static void FillBuffer1st() { FillBuffer(0); }
        static void FillBuffer2nd() { FillBuffer(bufferSize_/2); }

        // Called form the functions in "BSP_AudioIn_Overwrite.cpp"
        // and "BSP_AudioOut_Overwrite.cpp" 
        static void ErrorTrap();

    private:
        const int FS_;
        const InOutBoth IOBOTH_; 

        Array<uint16_t> inBuffer_;
        static Array<uint16_t> outBuffer_;
        static Array<uint16_t> tmp_;       

        __IO int32_t inIndex_;
        __IO int32_t tmpIndex_;

        static int32_t nData_;
        static int32_t bufferSize_;
        static __IO bool captured_;
        static __IO int32_t inOffset_;
        static __IO bool xferred_;

        void InitCodec(uint16_t inputDevice);
        void InitInput(uint16_t inputDevice);
        void SetInput();
        void SetOutput();
        void ClearBuffer();

        // This function pointer is assigned by
        // InputNormal() or InputReversal()
        void (SaiIO::*InputFp)(int16_t &, int16_t &);
        // For line input
        void InputNormal(int16_t &xL, int16_t &xR);
        // For MEMS microphone input
        void InputReversal(int16_t &xL, int16_t &xR);
        
        static void Captured(int32_t offset);
        static void FillBuffer(uint32_t offset);
    };
}
#endif  // F746_SAI_IO_HPP
