//------------------------------------------------
//  Realtime spectrum analyzer
//      Input: MEMS microphone or CN11
//
//  2016/04/19, Copyright (c) 2016 MIKAMI, Naoki
//------------------------------------------------

#include "main.h"

using namespace Mikami;

int16_t audio_in_buffer[BufferSize];
__IO bool audio_in_buffer_captured = false;
__IO int32_t audio_in_buffer_offset = 0;
__IO int32_t audio_in_buffer_length = 0;

// Switching input device
void SwitchInputDevice(int sw)
{
    uint16_t dev = (sw == 0) ? INPUT_DEVICE_DIGITAL_MICROPHONE_2
                             : INPUT_DEVICE_INPUT_LINE_1;
    InitRunAudioIn(dev, I2S_AUDIOFREQ_16K);
}

int main()
{
    const int FS = 8000;        // Sampling frequency: 8 kHz
    const int N_DATA = N_DATA_; // Number of data to be analyzed
    const int N_FFT = 512;      // Number of date for FFT

    const int X_WAV = 44;   // Origin for x axis of waveform
    const int Y_WAV = 36;   // Origin for y axis of waveform
    const int X0    = 40;   // Origin for x axis of spectrum
    const int Y0    = 234;  // Origin for y axis of spectrum
    const float DB1 = 2.4f; // Pixels for 1 dB
    const int BIN   = 1;    // Pixels per bin
    const int W_DB  = 60;   // Range in dB to be displayed
    const int X0_BTN = 340; // Holizontal position for left of buttons

    const uint32_t BACK_COLOR          = 0xFF006A6C;    // Teal green
    const uint32_t INACTIVE_COLOR      = BACK_COLOR & 0xD0FFFFFF;
    const uint32_t TOUCHED_COLOR       = 0xFF7F7FFF;
    const uint32_t ORIGINAL_COLOR      = 0xFF0068B7;
    const uint32_t INACTIVE_TEXT_COLOR = LCD_COLOR_GRAY;
    const uint32_t AXIS_COLOR          = 0xFFCCFFFF;
    const uint32_t LINE_COLOR          = LCD_COLOR_CYAN;

    LCD_DISCO_F746NG lcd;           // Object for LCD display
    TS_DISCO_F746NG ts;             // Object for touch pannel

    lcd.Clear(BACK_COLOR);

    // Setting of button group
    const string RUN_STOP[2] = {"RUN", "STOP"};
    ButtonGroup runStop(lcd, ts, X0_BTN, 10, 60, 40,
                        ORIGINAL_COLOR, BACK_COLOR,
                        2, RUN_STOP, 5, 0, 2);
    runStop.Draw(1, INACTIVE_COLOR, INACTIVE_TEXT_COLOR);

    const string MIC_LINE[2] = {"MIC", "LINE"};
    ButtonGroup micLine(lcd, ts, X0_BTN, 65, 60, 40,
                        ORIGINAL_COLOR, BACK_COLOR,
                        2, MIC_LINE, 5, 0, 2);
    micLine.DrawAll(INACTIVE_COLOR, INACTIVE_TEXT_COLOR);

    const string METHOD[3] = {"FFT", "Linear Pred.", "Cepstrum"};
    ButtonGroup method(lcd, ts, X0_BTN, 120, 125, 40,
                       ORIGINAL_COLOR, BACK_COLOR,
                       3, METHOD, 0, 5, 1);
    method.DrawAll(INACTIVE_COLOR, INACTIVE_TEXT_COLOR);
    // End of button group setting

    WaveformDisplay waveDisp(lcd, X_WAV, Y_WAV, N_DATA, 9,
                             AXIS_COLOR, LINE_COLOR, BACK_COLOR);

    SpectrumDisplay disp(lcd, N_FFT, X0, Y0, -45, DB1, BIN, W_DB, FS,
                         AXIS_COLOR, LINE_COLOR, BACK_COLOR);
    // Linear prediction:   order = 10
    // Cepstral smoothing:  highest quefrency = 40
    Selector analyzer(disp, N_DATA, N_FFT, 10, 40);

    // Initialize the infinite loop procedure
    int select = -1;
    int sw = 0; // 0: mic, 1: line
    int stop = 0;
    int16_t sn[N_DATA];

    // Wait for "RUN" button touched
    while (!runStop.Touched(0, TOUCHED_COLOR)) {}

    // Start of spectrum analyzing
    method.DrawAll(ORIGINAL_COLOR);

    while (true)
    {
        if (runStop.Touched(0, TOUCHED_COLOR))
        {
            SwitchInputDevice(sw);
            micLine.Draw(sw, TOUCHED_COLOR);
            micLine.Redraw(1-sw);
        }

        runStop.GetTouchedNumber(stop, TOUCHED_COLOR);
        if (stop == 0)
        {
            if (micLine.GetTouchedNumber(sw, TOUCHED_COLOR))
                SwitchInputDevice(sw);

            if (audio_in_buffer_captured)
            {
                // Waveform display
                int32_t index = audio_in_buffer_offset;     // right mic. or left line
//                int32_t index = audio_in_buffer_offset + 1;     // left mic. or right line
                Decimate(N_DATA, index, audio_in_buffer, sn);
                waveDisp.Execute(sn);
                // Spectrum analysis and display
                analyzer.Execute(sn, select);
                audio_in_buffer_captured = false;
            }
            else
                if (method.GetTouchedNumber(select, TOUCHED_COLOR))
                    analyzer.Execute(sn, select);
        }            
        else
        {
            if (runStop.Touched(1))
            {
                StopAudioIn();  // Stop sampling
                micLine.DrawAll(INACTIVE_COLOR, INACTIVE_TEXT_COLOR);
            }
            if (method.GetTouchedNumber(select, TOUCHED_COLOR))
                analyzer.Execute(sn, select);
        }
    }
}
