//-------------------------------------------------------
//  Class for spectrum analysis using cepstrum
//
//  2015/12/15, Copyright (c) 2015 MIKAMI, Naoki
//-------------------------------------------------------

#include "CepstrumAnalysis.hpp"

namespace Mikami
{
    CepstrumAnalyzer::CepstrumAnalyzer(int nData, int nFft, int nLifter)
            : AnalyzerBase(nData, nFft, nFft), N_LIFTER_(nLifter),
              yFft_(new Complex[nFft/2+1]), lifter_(new float[nLifter]),
              db_(new Complex[nFft/2+1]), cep_(new float[nFft]),
              cepLft_(new float[nFft]), cepFt_(new Complex[nFft/2+1])
    {
        // Generate lifter
        float pi2N = 3.1415926536f/nLifter;
        for (int n=0; n<nLifter; n++)
            lifter_[n] = 0.54f + 0.46f*cosf(pi2N*n);
    }

    CepstrumAnalyzer::~CepstrumAnalyzer()
    {
        delete[] yFft_;
        delete[] lifter_;
        delete[] db_;
        delete[] cep_;
        delete[] cepLft_;
        delete[] cepFt_;
    }

    void CepstrumAnalyzer::Analyze(const float xn[], float yn[])
    {
        fft_.Execute(xn, yFft_);    // Execute FFT

        // Translate to dB and to complex
        for (int n=0; n<=N_FFT_/2; n++)
            db_[n] = Complex(10.0f*log10f(Norm(yFft_[n])), 0);

        fft_.ExecuteIfft(db_, cep_);    // To cepstrum
        
        // Liftering
        cepLft_[0] = cep_[0];
        for (int n=1; n<N_LIFTER_; n++)
            cepLft_[n] = 2.0f*lifter_[n]*cep_[n];
        for (int n=N_LIFTER_; n<N_FFT_; n++)
            cepLft_[n] = 0.0f;
            
        // Smoothed spectrum
        fft_.Execute(cepLft_, cepFt_);
        for (int n=0; n<=N_FFT_/2; n++)
            yn[n] = cepFt_[n].real();
    }
}
