//-----------------------------------------------------------
//  Class for waveform display
//
//  2015/12/15, Copyright (c) 2015 MIKAMI, Naoki
//-----------------------------------------------------------

#ifndef F746_WAVEFORM_DISPLAY_HPP
#define F746_WAVEFORM_DISPLAY_HPP

#include "mbed.h"

namespace Mikami
{
    class WaveformDisplay
    {
    public:
        WaveformDisplay(LCD_DISCO_F746NG &lcd,
                        uint16_t x0, uint16_t y0, int nData,
                        uint16_t rShift,
                        uint32_t axisColor, uint32_t lineColor,
                        uint32_t backColor)
            : X0_(x0), Y0_(y0), N_DATA_(nData), R_SHIFT_(rShift),
              AXIS_COLOR_(axisColor), LINE_COLOR_(lineColor),
              BACK_COLOR_(backColor), lcd_(lcd) { Axis(); }
        
        void Execute(const int16_t xn[])
        {
            static const uint16_t LIMIT1 = Y0_ + LIMIT2_;
            static const uint16_t LIMIT2 = Y0_ - LIMIT2_;
            Axis();
            lcd_.SetTextColor(LINE_COLOR_);
            uint16_t x1 = X0_;
            uint16_t y1 = Clip(xn[0]);
            for (int n=1; n<N_DATA_; n++)
            {
                uint16_t x2 = X0_ + n;
                uint16_t y2 = Clip(xn[n]);
                if ( ((y2 == LIMIT1) && (y1 == LIMIT1)) ||
                     ((y2 == LIMIT2) && (y1 == LIMIT2)) )
                {   // Out of displaying boundaries
                    lcd_.SetTextColor(LCD_COLOR_RED);
                    lcd_.DrawHLine(x1, y1, 1);
                    lcd_.SetTextColor(LINE_COLOR_);
                }
                else
                    lcd_.DrawLine(x1, y1, x2, y2);
                if ((y1 == LIMIT1) || (y1 == LIMIT2))
                    lcd_.DrawPixel(x1, y1, LCD_COLOR_RED);
                x1 = x2;
                y1 = y2;
            }
            lcd_.SetTextColor(BACK_COLOR_);
        }
        
    private:
        const uint16_t X0_;
        const uint16_t Y0_;
        const int N_DATA_;
        const uint16_t R_SHIFT_;
        const uint32_t AXIS_COLOR_;
        const uint32_t LINE_COLOR_;
        const uint32_t BACK_COLOR_;
        static const int LIMIT_ = 32;
        static const int LIMIT2_ = LIMIT_ + 1;
        
        LCD_DISCO_F746NG& lcd_;
        
        // Clipping
        uint16_t Clip(int16_t xn)
        {
            int16_t x = xn >> R_SHIFT_;
            if (x >  LIMIT_ ) x =  LIMIT2_;
            if (x < -LIMIT_ ) x = -LIMIT2_ ;
            return Y0_ - x;
        }
        
        void Axis()
        {
            lcd_.SetTextColor(BACK_COLOR_);
            lcd_.FillRect(X0_, Y0_-LIMIT2_, N_DATA_, LIMIT2_*2+1);

            lcd_.SetTextColor(AXIS_COLOR_);
            lcd_.DrawLine(X0_-5, Y0_, X0_+N_DATA_+5, Y0_);
        }        

        // disallow copy constructor and assignment operator
        WaveformDisplay(const WaveformDisplay& );
        WaveformDisplay& operator=(const WaveformDisplay& );
    };
}
#endif  // F746_WAVEFORM_DISPLAY_HPP
