//-------------------------------------------------------
//  Class for display spectrum
//
//  2015/12/11, Copyright (c) 2015 MIKAMI, Naoki
//-------------------------------------------------------

#include "SpectrumDisplay.hpp"

namespace Mikami
{
    SpectrumDisplay::SpectrumDisplay(
            LCD_DISCO_F746NG &lcd,
            int nFft, int x0, int y0,
            float db1, int bin, float maxDb, int fs,
            uint32_t axisColor, uint32_t lineColor,
            uint32_t backColor)
        : N_FFT_(nFft), X0_(x0), Y0_(y0),
          DB1_(db1), BIN_(bin), MAX_DB_(maxDb),
          FS_(fs), AXIS_COLOR_(axisColor),
          LINE_COLOR_(lineColor), BACK_COLOR_(backColor),
          lcd_(lcd)
    {
        AxisX();
        AxisY();
    }

    void SpectrumDisplay::Draw(float db[])
    {
        const float OFFSET = -50.0f;
        lcd_.SetTextColor(BACK_COLOR_);
        lcd_.FillRect(X0_+1, Y0_-(int)(MAX_DB_*DB1_),
                      N_FFT_*BIN_/2, MAX_DB_*DB1_);

        float h = ((db[1] + OFFSET) > 0)? db[1] + OFFSET : 0;
        int y1 = Y0_ - (int)(h*DB1_);
        for (int n=1; n<N_FFT_/2; n++)
        {
            float h2 = ((db[n+1] + OFFSET) > 0)? db[n+1] + OFFSET : 0;
            if (h2 > MAX_DB_) h2 = MAX_DB_;
            int y2 = Y0_ - (int)(h2*DB1_);
            lcd_.SetTextColor(LINE_COLOR_);
            lcd_.DrawLine(X0_+n, y1, X0_+n+1, y2);
            y1 = y2;
        }
        lcd_.SetTextColor(AXIS_COLOR_);
        lcd_.DrawLine(X0_, Y0_, X0_+BIN_*N_FFT_/2, Y0_);
    }

    // Clear Spectrum
    void SpectrumDisplay::Clear()
    {
        lcd_.SetTextColor(BACK_COLOR_);
        lcd_.FillRect(X0_+1, Y0_-(int)(MAX_DB_*DB1_),
                      N_FFT_*BIN_/2, MAX_DB_*DB1_);
    }

    // x-axis
    void SpectrumDisplay::AxisX()
    {
        lcd_.SetFont(&Font12);

        lcd_.SetTextColor(AXIS_COLOR_);
        lcd_.DrawLine(X0_, Y0_, X0_+BIN_*N_FFT_/2, Y0_);
        float dx = BIN_*(N_FFT_*1000.0f)/(float)FS_;
        for (int n=0; n<=(FS_/1000)/2; n++)
        {
            int xTick = X0_ + (int)(dx*n + 0.5f);
            char s[5];
            sprintf(s, "%d", n);
            DrawString(xTick-3, Y0_+10, s);
            lcd_.DrawLine(xTick, Y0_, xTick, Y0_+5);
        }
        DrawString(X0_+74, Y0_+24, "Frequency [kHz]");
    }

    // y-axis
    void SpectrumDisplay::AxisY()
    {
        lcd_.SetFont(&Font12);
        
        lcd_.SetTextColor(AXIS_COLOR_);
        lcd_.DrawLine(X0_, Y0_+5, X0_, Y0_-(int)(MAX_DB_*DB1_));
        for (int n=0; n<=(int)MAX_DB_; n+=20)
        {
            int yTick = Y0_-(int)(DB1_*n);
            char s[5];
            sprintf(s, "%3d", n);
            DrawString(X0_-30, yTick-5, s);
            lcd_.DrawLine(X0_-5, yTick, X0_, yTick);
        }
        DrawString(X0_-27, Y0_-(int)(DB1_*MAX_DB_)-18, "[dB]");
    }
}
