//------------------------------------------------
//  Realtime spectrum analyzer
//      Input: A0 (CN5)
//
//  2016/02/22, Copyright (c) 2016 MIKAMI, Naoki
//------------------------------------------------

#include "button_group.hpp"
#include "Sampler.hpp"
#include "WaveformDisplay.hpp"
#include "AnalysisSelector.hpp"
using namespace Mikami;

int main()
{
    const int FS = 10000;   // Sampling frequency: 10 kHz
    const int N_DATA = 260; // Number of data to be analyzed
    const int N_FFT = 512;  // Number of date for FFT

    const int X_WAV = 44;   // Origin for x axis of waveform
    const int Y_WAV = 36;   // Origin for y axis of waveform
    const int X0    = 40;   // Origin for x axis of spectrum
    const int Y0    = 234;  // Origin for y axis of spectrum
    const float DB1 = 2.4f; // Pixels for 1 dB
    const int BIN   = 1;    // Pixels per bin
    const int W_DB  = 60;   // Range in dB to be displayed
    const int X0_BTN = 340; // Holizontal position for left of buttons

    const uint32_t BACK_COLOR          = 0xFF006A6C;    // Teal green
    const uint32_t INACTIVE_COLOR      = BACK_COLOR & 0xD0FFFFFF;
    const uint32_t TOUCHED_COLOR       = 0xFF7F7FFF;
    const uint32_t ORIGINAL_COLOR      = 0xFF0068B7;
    const uint32_t INACTIVE_TEXT_COLOR = LCD_COLOR_GRAY;
    const uint32_t AXIS_COLOR          = 0xFFCCFFFF;
    const uint32_t LINE_COLOR          = LCD_COLOR_CYAN;

    LCD_DISCO_F746NG lcd;           // Object for LCD display
    TS_DISCO_F746NG ts;             // Object for touch pannel
    Sampler input(A0, FS, N_DATA);  // Object for sampling
    int16_t* sn = input.Get();      // Input data

    lcd.Clear(BACK_COLOR);

    // Setting of button group
    const string RUN_STOP[2] = {"RUN", "STOP"};
    ButtonGroup runStop(lcd, ts, X0_BTN, 10, 60, 40,
                        ORIGINAL_COLOR, BACK_COLOR,
                        2, RUN_STOP, 5, 0, 2);
    runStop.Draw(1, INACTIVE_COLOR, INACTIVE_TEXT_COLOR);

    const string NORM_INV[2] = {"NORM", "INV"};
    ButtonGroup normInv(lcd, ts, X0_BTN, 65, 60, 40,
                        ORIGINAL_COLOR, BACK_COLOR,
                        2, NORM_INV, 5, 0, 2);
    normInv.DrawAll(INACTIVE_COLOR, INACTIVE_TEXT_COLOR);

    const string METHOD[3] = {"FFT", "Linear Pred.", "Cepstrum"};
    ButtonGroup method(lcd, ts, X0_BTN, 120, 125, 40,
                       ORIGINAL_COLOR, BACK_COLOR,
                       3, METHOD, 0, 5, 1);
    method.DrawAll(INACTIVE_COLOR, INACTIVE_TEXT_COLOR);
    // End of button group setting

    WaveformDisplay waveDisp(lcd, X_WAV, Y_WAV, N_DATA, 10,
                             AXIS_COLOR, LINE_COLOR, BACK_COLOR);

    SpectrumDisplay disp(lcd, N_FFT, X0, Y0, DB1, BIN, W_DB, FS,
                         AXIS_COLOR, LINE_COLOR, BACK_COLOR);
    // Linear prediction:   order = 14
    // Cepstral smoothing:  highest quefrency = 50
    Selector analyzer(disp, N_DATA, N_FFT, 14, 50);

    // Wait for "RUN" button touched
    while (!runStop.Touched(0, TOUCHED_COLOR)) {}

    // Start of spectrum analyzing
    method.DrawAll(ORIGINAL_COLOR);
    int inv = 0;    // 0: "NORM", 1: "INV"
    int select = -1;
    
    while (true)
    {
        if (runStop.Touched(0))
        {
            input.Start(inv == 1);
            normInv.Draw(inv, TOUCHED_COLOR);
            normInv.Redraw(1-inv);
        }

        int stop;
        runStop.GetTouchedNumber(stop, TOUCHED_COLOR);
        if (stop == 0)
        {
            normInv.GetTouchedNumber(inv, TOUCHED_COLOR);

            if (input.Filled())
            {
                // Restart
                input.Start(inv == 1);

                // Waveform display
                waveDisp.Execute(sn);
                // Spectrum analysis and display
                analyzer.Execute(sn, select);
            }
            else
                if (method.GetTouchedNumber(select, TOUCHED_COLOR))
                    analyzer.Execute(sn, select);
        }            
        else
        {
            if (runStop.Touched(1))
            {
                input.Stop();   // Stop timer interrupt
                normInv.DrawAll(INACTIVE_COLOR, INACTIVE_TEXT_COLOR);
            }
            if (method.GetTouchedNumber(select, TOUCHED_COLOR))
                analyzer.Execute(sn, select);
        }
    }
}
