//-----------------------------------------------------------
//  NumericUpDown class
//      指定された桁を up/down する GUI 部品
//      6 桁まで可能
//
//  2017/08/21, Copyright (c) 2017 MIKAMI, Naoki
//-----------------------------------------------------------

#include "NumericUpDown.hpp"

namespace Mikami
{
    // digit    表示する桁数
    // initVal  最初に表示する数値
    NumericUpDown::NumericUpDown(uint16_t digit, uint16_t x0, uint16_t y0,
                                 uint32_t initVal, uint32_t max, uint32_t min,
                                 string unit, float delay)
        : DIGIT_(digit), X0_(x0), Y0_(y0), MAX_(max), MIN_(min),
          POW10_(6, (uint32_t []){1, 10, 100, 1000, 10000, 10000}),
          numUd_(-1), value_(initVal),
          delay_(delay), labels_(digit), lcd_(GuiBase::GetLcd())
    {
        MBED_ASSERT(DIGIT_ <= 6);   // 6 桁まで
        sprintf(fmt_, "%%""0%1dd", DIGIT_);
        for (int n=0; n<DIGIT_; n++)
            labels_[n] = new Label(
                X0_+10+n*XW_, Y0_-46, "", Label::LEFT, Font16);
        Set(initVal);       // 初期値として表示
        Label lbUnit(X0_+0+DIGIT_*XW_, Y0_-46, unit, Label::LEFT, Font16);
        Array<string> nonString(DIGIT_*2, "");
        ud_ = new ButtonGroup(X0_, Y0_-81, 30, 30, DIGIT_*2, nonString,
                              XW_-30, 22, DIGIT_);
        for (int n=0; n<DIGIT_*2; n++) UpDownSymbol(X0_, Y0_, n);
    }

    // Up, Down をタッチした場合は対応する数値を +1 または -1 する
    bool NumericUpDown::Touched()
    {
        if (!delay_.IsEnabled()) return false;
        int num;
        if (!ud_->GetTouchedNumber(num)) return false;

        // タッチ後一定の時間が経過してから，再びタッチの検出を有効にするため
        delay_.Disable();

        // 三角を再描画
        UpDownSymbol(X0_, Y0_, num);
        if (numUd_ != -1) UpDownSymbol(X0_, Y0_, numUd_);
        numUd_ = num;

        int index = num % DIGIT_;
        if (num < DIGIT_) value_ += POW10_[DIGIT_-index-1];
        else
            if (index != 0)
                value_ -= POW10_[DIGIT_-index-1];
            else
                if (value_ >= POW10_[DIGIT_-1])
                    value_ -= POW10_[DIGIT_-index-1];
        Set(value_);
        return true;
    }

    // 引数の数値を Label に表示する
    void NumericUpDown::Set(uint32_t frq)
    {
        if (frq > MAX_) frq = MAX_;
        if (frq < MIN_) frq = MIN_;
        char frqCh[7];
        sprintf(frqCh, fmt_, frq);

        for (int n=0; n<DIGIT_; n++)
            labels_[n]->Draw((string)""+frqCh[n]);
        value_ = frq;
    }

    // 数値の up, down に使う三角形の描画
    void NumericUpDown::UpDownSymbol(uint16_t x, uint16_t y, int k)
    {
        lcd_.SetTextColor(GuiBase::ENUM_TEXT);
        Point *pt = k < DIGIT_ ? ptU_ : ptD_;
        int16_t y0 = k < DIGIT_ ? y - 25 : y + 31;

        Point tri[3];
        for (int n=0; n<3; n++)
        {
            tri[n].X = pt[n].X + x + XW_*(k % DIGIT_) + 15;
            tri[n].Y = pt[n].Y + y0 - 46;
        }
        lcd_.FillPolygon(tri, 3);
    }
    
    Point NumericUpDown::ptU_[] = {{0, 0}, {-SIZE_, SIZE_}, {SIZE_, SIZE_}};
    Point NumericUpDown::ptD_[] = {{0, SIZE_}, {SIZE_, 0}, {-SIZE_, 0}};
}
