//--------------------------------------------------------------
//  ヒルベルト変換フィルタを使う周波数シフタ
//
//  2018/03/19, Copyright (c) 2018 MIKAMI, Naoki
//--------------------------------------------------------------

#ifndef HILBERT_FREQUENCY_CONVERTER_HPP
#define HILBERT_FREQUENCY_CONVERTER_HPP

#include "ProcessingBase.hpp"
#include "Biquad.hpp"               // 直流分除去フィルタ用
#include "DC_Cut_Coefficients.hpp"  // 直流分除去フィルタの係数
#include "HilbertTransform.hpp"     // ヒルベルト変換用フィルタ
#include "coefsHilbert162.hpp"      // ヒルベルト変換用フィルタの係数

namespace Mikami
{
    class FrqShifterHilbert : public ProcessingBase
    {
    public:
        FrqShifterHilbert(float fS, float fShift = 0)
            : dcCutFilter_(dcCut_), hilbertFIR_(ORDER_HILBERT_, hmHilbert_),
              FS_(fS), PI2_(3.1415926536f*2),
              phi_(0), dPhi_(fShift*PI2_/FS_)
        {   SetFrequency(fShift); }

        virtual float Execute(float xn)
        {
            float un = dcCutFilter_.Execute(xn*dcCutG0_);   // 直流分除去フィルタ
            float yI, yQ;
            hilbertFIR_.Execute(un, yI, yQ);                // ヒルベルト変換用フィルタ

            float yn = yI*cosf(phi_) - yQ*sinf(phi_);

            phi_ = phi_ + dPhi_;
            if (phi_ > PI2_) phi_ = phi_ - PI2_;

            return yn;
        }

        // 周波数のシフト量を設定
        void SetFrequency(float fShift)
        {   dPhi_ = fShift*PI2_/FS_; }

    private:
        //
        Biquad dcCutFilter_;    // 直流分除去フィルタ
        Hilbert hilbertFIR_;    // ヒルベルト変換用フィルタ

        const float FS_;
        const float PI2_;

        float phi_, dPhi_;

        // disallow copy constructor and assignment operator
        FrqShifterHilbert(const FrqShifterHilbert& );
        FrqShifterHilbert& operator=(const FrqShifterHilbert& );
    };
}
#endif  // HILBERT_FREQUENCY_CONVERTER_HPP
