//--------------------------------------------------------------
//  SD_BinaryReader class
//      SD カードの内容を読み出す
//          最初の4バイト：データサイズ
//          それ以降： int16_t のデータ
//
//  2016/04/07, Copyright (c) 2016 MIKAMI, Naoki
//--------------------------------------------------------------

#ifndef SD_BINARY_READER_HPP
#define SD_BINARY_READER_HPP

#include "SDFileSystem.h"
#include "BlinkLabel.hpp"
#include <string>

namespace Mikami
{
    class SD_BinaryReader
    {
    public:
        SD_BinaryReader() : STR_("sd"), ok_(false)
        {
            sd_ = new SDFileSystem(STR_.c_str());
            sd_->mount();
        }

        ~SD_BinaryReader()
        {
            sd_->unmount();
            delete sd_;
        }
        
        void Open(const string fileName)
        {
            string name = (string)"/" + STR_ + "/" + fileName;
            fp_ = fopen(name.c_str(), "rb");
            if (fp_ == NULL) ErrorMsg("open error!!");
        }
        
        void Close() { fclose(fp_); }
        
        // ファイルからデータサイズの読み出し
        //      戻り値： int16_t 型のデータサイズ
        int32_t ReadSize()
        {
            fread(&size_, sizeof(int), 1, fp_);
            ok_ = true;
            return size_;
        }

        // ファイルからデータの取得
        void Read(int16_t data[], uint32_t size)
        {
            if (!ok_) ErrorMsg("Get data FAILED");
            fread(data, sizeof(int16_t), size, fp_);
        }        

        // データサイズの取得
        //      戻り値： int16_t 型のデータサイズ
        int32_t GetSize()
        {
            if (!ok_) ErrorMsg("Get data size FAILED");
            return size_;
        }

    private:
        const string STR_;
        
        SDFileSystem *sd_;
        FILE *fp_;
        
        bool ok_;
        int32_t size_;  // word count, word = int16_t
        
        void ErrorMsg(char msg[])
        {
            BlinkLabel errLabel(240, 100, msg, Label::CENTER);
        }

        // disallow copy constructor and assignment operator
        SD_BinaryReader(const SD_BinaryReader&);
        SD_BinaryReader& operator=(const SD_BinaryReader&);
    };
}
#endif  // SD_BINARY_READER_HPP
