//-----------------------------------------------------------
//  周波数が対数スケールの周波数特性を描画するクラス
//  FrqRespDrawer class Header
//
//  2016/04/17, Copyright (c) 2016 MIKAMI, Naoki
//-----------------------------------------------------------

#ifndef F746_FRQ_RESP_DRAWER_HPP
#define F746_FRQ_RESP_DRAWER_HPP

#include "NumericLabel.hpp"
#include "FrequancyResponseBase.hpp"

namespace Mikami
{
    class FrqRespDrawer
    {
    public:
        // Constructor
        FrqRespDrawer(uint16_t org, float min, float max, uint16_t dec,
                      uint16_t orgY, float minDb, float maxDb, uint16_t db10,
                      float fs,
                      uint32_t lineColor = 0xFF00B0FF,
                      uint32_t axisColor = LCD_COLOR_LIGHTGRAY,
                      uint32_t backColor = GuiBase::ENUM_BACK)
            : lcd_(GuiBase::GetLcdPtr()),
              ORG_(org), MIN_(min), MAX_(max), DEC_(dec),
              ORGY_(orgY), MIN_DB_(minDb), MAX_DB_(maxDb), DB10_(db10),
              FS_(fs),
              LINE_COLOR_(lineColor),
              AXIS_COLOR_(axisColor),
              BACK_COLOR_(backColor),
              DB1_(db10*0.1f) {}

        // 周波数に対応する x 座標値の取得
        int X(float frq)
        {   return Round(ORG_ + DEC_*log10f(frq/MIN_));  }

        // x 座標値を周波数に変換
        float PosToFrq(uint16_t x)
        {   return MIN_*powf(10.0f, (x - ORG_)/(float)DEC_); }

        // 目盛線の描画
        void DrawAxis();

        // 横軸の目盛値の表示用
        void DrawCharX(uint32_t frq, int offsetY,
                          const char str[], sFONT &fonts = Font12,
                          uint32_t textColor = LCD_COLOR_WHITE)
        {   Label frequency(X(frq), ORGY_+offsetY, str, Label::CENTER); }

        // 縦軸の目盛値の表示
        void DrawNumericY(int offsetX, int offsetY, int count,
                          uint16_t d_dB, const char fmt[], sFONT &fonts = Font12,
                          uint32_t textColor = LCD_COLOR_WHITE); 

        // 周波数特性のグラフの描画
        void DrawGraph(FrequencyResponse &frqResp, uint32_t color);
        void DrawGraph(FrequencyResponse &frqResp)
        {   DrawGraph(frqResp, LINE_COLOR_);}

        // 消去
        void Erase();

    private:
        LCD_DISCO_F746NG *lcd_;
        const uint16_t ORG_;    // 横軸の目盛の最小値に対応する位置
        const float MIN_;       // 横軸の目盛の最小値
        const float MAX_;       // 横軸の目盛の最大値
        const uint16_t DEC_;    // 周波数の 10 倍に対応する長さ (pixels)
        const uint16_t ORGY_;   // 縦軸の目盛の最小値に対応する位置
        const float MIN_DB_;    // 縦軸の目盛の最小値 [dB]
        const float MAX_DB_;    // 縦軸の目盛の最大値 [dB]
        const uint16_t DB10_;   // 10 dB 対応する長さ (pixels)
        const float FS_;        // 標本化周波数
        const uint32_t LINE_COLOR_;
        const uint32_t AXIS_COLOR_;
        const uint32_t BACK_COLOR_;
        const float DB1_;
        
        // 丸め
        int Round(float x) { return x + 0.5f - (x < 0); }  

        // 10 のべき乗かどうかの検査
        bool PowersOf10(float  x)
        {   return fabsf(log10f(x) - Round(log10f(x))) < 0.01f; }

        // disallow copy constructor and assignment operator
        FrqRespDrawer(const FrqRespDrawer&);
        FrqRespDrawer& operator=(const FrqRespDrawer&);
    };
}
#endif  // F746_FRQ_RESP_DRAWER_HPP
