//--------------------------------------------------------------
//  FileSelector class
//      SD カード内のファイル名の一覧を表示し，ファイルを選択する
//
//  2016/04/10, Copyright (c) 2016 MIKAMI, Naoki
//--------------------------------------------------------------

#ifndef FILE_SELECTOR_HPP
#define FILE_SELECTOR_HPP

#include "mbed.h"
#include "Label.hpp"
#include "ButtonGroup.hpp"
#include "SD_BinaryReader.hpp"
#include "SDFileSystem.h"
#include <algorithm>    // sort() で使用
#include <string>

namespace Mikami
{
    class FileSelector
    {
    public:
        FileSelector(uint8_t x0, uint8_t y0, int maxFiles,
                     int maxNameLength, SD_BinaryReader &reader)
            : X_(x0), Y_(y0), W_H_(24), V_L_(36),
              MAX_FILES_(maxFiles), MAX_NAME_LENGTH_(maxNameLength),
              BASE_COLOR_(0xFF80FFA0), TOUCHED_COLOR_(0xFF80FFFF),
              fileNames_(new string[maxFiles]),
              sortedFileNames_(new string[maxFiles]),
              lcd_(GuiBase::GetLcdPtr()),
              sdReader_(reader), prev_(-1) {}

        ~FileSelector()
        {
            for (int n=0; n<fileCount_; n++)
                delete fileNameLabels_[n];
            delete[] fileNameLabels_;
            delete[] rect_;
            delete[] nonString_;
            delete[] sortedFileNames_;
            delete[] fileNames_;
        }
        bool CreateTable()
        {
            DIR* dp = opendir("/sd");
            fileCount_ = 0; 
            if (dp != NULL)
            {
                dirent* entry;
                for (int n=0; n<256; n++)
                {
                    entry = readdir(dp);
                    if (entry == NULL) break;

                    string strName = entry->d_name;
                    if ( (strName.find(".bin") != string::npos) ||
                         (strName.find(".BIN") != string::npos) )
                    {
                        sdReader_.Open(strName);        // ファイルオープン
                        // 小さすぎるファイルは除外
                        if ((sdReader_.ReadSize()) > 4096)
                        {
                            fileNames_[fileCount_] = strName;
                            fileCount_++;
                        }
                        sdReader_.Close();
                    }
                    if (fileCount_ >= MAX_FILES_) break;
                }
                closedir(dp); 
            }
            else
                return false;
                
            if (fileCount_ == 0) return false;

            nonString_ = new string[fileCount_];
            for (int n=0; n<fileCount_; n++) nonString_[n] = "";
            rect_ = new ButtonGroup(X_, Y_, W_H_, W_H_, fileCount_,
                                    nonString_, 0, V_L_-W_H_, 1,
                                    -1, Font12, 0, GuiBase::ENUM_BACK,
                                    BASE_COLOR_, TOUCHED_COLOR_);
            for (int n=0; n<fileCount_; n++) rect_->Erase(n);
            CreateLabels();   

            return true;
        }

        // ファイルを選択する
        bool Select(string &fileName)
        {
            int n;
            if (rect_->GetTouchedNumber(n))
            {
                fileNameLabels_[n]->Draw(GetFileNameNoExt(n), TOUCHED_COLOR_);
                if ((prev_ >= 0) && (prev_ != n))
                    fileNameLabels_[prev_]->Draw(GetFileNameNoExt(prev_));
                prev_ = n;
                fileName = sortedFileNames_[n];
                return true;
            }
            else
                return false;
        }

        // ファイルの一覧の表示
        void DisplayFileList(bool sort = true)
        {
            for (int n=0; n<fileCount_; n++)
                sortedFileNames_[n] = fileNames_[n];
            if (sort)
                std::sort(sortedFileNames_, sortedFileNames_+fileCount_); 
            
            Erase(MAX_NAME_LENGTH_*((sFONT *)(&Font16))->Width, 270-Y_);
            rect_->DrawAll();
            for (int n=0; n<fileCount_; n++)
                fileNameLabels_[n]->Draw(GetFileNameNoExt(n));
        }

        void Erase(uint16_t width, uint16_t height,
                   uint32_t color = GuiBase::ENUM_BACK)
        {
            lcd_->SetTextColor(color);
            lcd_->FillRect(X_, Y_, width, height);
        }

    private:
        const uint8_t X_, Y_, W_H_, V_L_;
        const int MAX_FILES_;
        const int MAX_NAME_LENGTH_;
        const uint32_t BASE_COLOR_;
        const uint32_t TOUCHED_COLOR_;
        
        string *fileNames_;
        string *sortedFileNames_;
        string *nonString_;
        ButtonGroup *rect_;
        Label **fileNameLabels_;
        LCD_DISCO_F746NG *lcd_;
        SD_BinaryReader &sdReader_;
        int fileCount_;
        int prev_;

        // Label を生成
        void CreateLabels()
        {
            fileNameLabels_ = new Label *[fileCount_+1];
            for (int n=0; n<fileCount_; n++)
                fileNameLabels_[n] = new Label(X_+30, Y_+5+V_L_*n, "",
                                               Label::LEFT, Font16,
                                               BASE_COLOR_);
        }

        // 拡張子を削除した文字列を取得
        string GetFileNameNoExt(int n)
        {
            string name = sortedFileNames_[n];
            name.erase(name.find("."));
            return name.substr(0, MAX_NAME_LENGTH_);
        }

        // disallow copy constructor and assignment operator
        FileSelector(const FileSelector&);
        FileSelector& operator=(const FileSelector&);
    };
}
#endif  // FILE_SELECTOR_HPP
