//-----------------------------------------------------------
//  FrqRespDrawer class (Abstract base class) -- Header
//
//  2016/03/30, Copyright (c) 2016 MIKAMI, Naoki
//-----------------------------------------------------------

#ifndef F746_FRQ_RESP_DRAWER_HPP
#define F746_FRQ_RESP_DRAWER_HPP

#include <complex>  // requisite for complex
#include "Label.hpp"

namespace Mikami
{
    typedef complex<float> Complex; // define "Complex"

    class FrqRespDrawer
    {
    public:
        struct Params
        {
            const uint16_t ORG;     // 横軸の目盛の最小値に対応する位置
            const float MIN;        // 横軸の目盛の最小値
            const float MAX;        // 横軸の目盛の最大値
            const uint16_t DEC;     // 周波数の 10 倍に対応する長さ (pixels)
            const uint16_t ORGY;    // 縦軸の目盛の最小値に対応する位置
            const float MIN_DB;     // 縦軸の目盛の最小値 [dB]
            const float MAX_DB;     // 縦軸の目盛の最大値 [dB]
            const uint16_t DB10;    // 10 dB 対応する長さ (pixels)
            const float FS;         // 標本化周波数
            const uint32_t LINE_COLOR;
            const uint32_t AXIS_COLOR;
            const uint32_t BACK_COLOR;
            
            Params(uint16_t org, float min, float max, uint16_t dec,
                   uint16_t orgY, float minDb, float maxDb, uint16_t db10,
                   float fs,
                   uint32_t lineColor = 0xFF00B0FF,
                   uint32_t axisColor = LCD_COLOR_LIGHTGRAY,
                   uint32_t backColor = GuiBase::ENUM_BACK)
                : ORG(org), MIN(min), MAX(max), DEC(dec),
                  ORGY(orgY), MIN_DB(minDb), MAX_DB(maxDb), DB10(db10),
                  FS(fs),
                  LINE_COLOR(lineColor),
                  AXIS_COLOR(axisColor),
                  BACK_COLOR(backColor) {}
        };
        
        // Constructor
        FrqRespDrawer(Params p)
            : lcd_(GuiBase::GetLcdPtr()), P_(p), DB1_(p.DB10*0.1f) {}

        // 周波数に対応する x 座標値の取得
        int X(float frq)
        {   return Round(P_.ORG + P_.DEC*log10f(frq/P_.MIN));  }

        // x 座標値を周波数に変換
        float PosToFrq(uint16_t x)
        { return P_.MIN*powf(10.0f, (x - P_.ORG)/(float)P_.DEC); }

        // 目盛線の描画
        void DrawAxis();

        // 横軸の目盛値の表示用
        void DrawCharX(uint32_t frq, int offsetY,
                          const char str[], sFONT &fonts = Font12,
                          uint32_t textColor = LCD_COLOR_WHITE)
        {   Label frequency(X(frq), P_.ORGY+offsetY, str, Label::CENTER); }

        // 縦軸の目盛値の表示
        void DrawNumericY(int offsetX, int offsetY, int count,
                          uint16_t d_dB, const char fmt[], sFONT &fonts = Font12,
                          uint32_t textColor = LCD_COLOR_WHITE); 

        // 周波数特性のグラフの描画
        void DrawGraph();
        
        // 周波数応答の絶対値を返す関数, 引数: z^(-1)
        virtual float AbsH_z(Complex u) = 0;

        // 消去
        void Erase();

    private:
        LCD_DISCO_F746NG *lcd_;
        const Params P_;
        const float DB1_;

        // 丸め
        int Round(float x) { return x + 0.5f - (x < 0); }  

        // 10 のべき乗かどうかの検査
        bool PowersOf10(float  x)
        { return fabsf(log10f(x) - Round(log10f(x))) < 0.01f; }

        // disallow copy constructor and assignment operator
        FrqRespDrawer(const FrqRespDrawer&);
        FrqRespDrawer& operator=(const FrqRespDrawer&);
    };
}
#endif  // F746_FRQ_RESP_DRAWER_HPP
