//--------------------------------------------------------------
//  QspiBinaryReader class
//      QSPI 接続のフラッシュメモリの内容を読み出す
//          最初の4バイト：データサイズ
//          それ以降： int16_t のモノラルデータ
//
//  2016/03/25, Copyright (c) 2016 MIKAMI, Naoki
//--------------------------------------------------------------

#ifndef QSPI_BINARY_READER_HPP
#define QSPI_BINARY_READER_HPP

#include "QSPI_DISCO_F746NG.h"

namespace Mikami
{
    class QspiBinaryReader
    {
    public:
        QspiBinaryReader()
        {
            // Check initialization
            if (qspi_.Init() != QSPI_OK)
                ErrorMsg("Initialization FAILED\r\n");
            else
                printf("Initialization PASSED\r\n");

            // Check memory informations
            QSPI_Info pQSPI_Info;
            qspi_.GetInfo(&pQSPI_Info);
            if ((pQSPI_Info.FlashSize          != N25Q128A_FLASH_SIZE) ||
                (pQSPI_Info.EraseSectorSize    != N25Q128A_SUBSECTOR_SIZE) || 
                (pQSPI_Info.ProgPageSize       != N25Q128A_PAGE_SIZE) ||
                (pQSPI_Info.EraseSectorsNumber != N25Q128A_SUBSECTOR_SIZE) ||
                (pQSPI_Info.ProgPagesNumber    != N25Q128A_SECTOR_SIZE))
                ErrorMsg("Get informations FAILED\r\n");
            else
                printf("Get informations PASSED\r\n");

            ok_ = false;
        }
        
        // データサイズの読み出し
        //      戻り値： int16_t 型のデータサイズ
        int32_t ReadSize()
        {
            uint8_t byte4[5];   // サイズを１バイト分大きくしない場合，実行時のエラーとなる．
            qspi_.Read(byte4, 0x0000, 4);
            size_ = ((uint32_t *)byte4)[0];
            ok_ = true;
            return size_;
        }
        
        // データサイズの取得
        //      戻り値： int16_t 型のデータサイズ
        int32_t GetSize()
        {
            if (!ok_) ErrorMsg("Get data size FAILED\r\n");
            return size_;
        }

        // データの取得
        void Read(int16_t data[], uint32_t offset, uint32_t size)
        {
            if (!ok_) ErrorMsg("Get data size FAILED\r\n");
            uint8_t *xn = (uint8_t *)data;
            qspi_.Read(xn, offset*2+4, size*2);
        }

    private:
        QSPI_DISCO_F746NG qspi_;
        bool ok_;
        int32_t size_;  // word count, word = int16_t
        
        void ErrorMsg(char msg[])
        {
            error(msg);
            while (true) {}
        }
    };
}
#endif  // QSPI_BINARY_READER_HPP
