//--------------------------------------------------------------
// 残響生成器で使うフィルタ
//
// 2018/07/11, Copyright (c) 2018 MIKAMI, Naoki
//--------------------------------------------------------------

#ifndef REVERB_UNIT_HPP
#define REVERB_UNIT_HPP

#include "mbed.h"

// 残響生成ユニットのための基底クラス
template<int delay> class Reverb
{
private:
    int ptr_;
    float un_[delay];    // for delay
protected:
    float Get() { return un_[ptr_]; }
    void Set(float x)
    {
        un_[ptr_] = x;   
        if (++ptr_ >=  delay) ptr_ = 0;
    }
public:
    // コンストラクタ
    Reverb() : ptr_(0) { Clear(); }

    // 残響生成フィルタの実行（純粋仮想関数）
    virtual float Execute(float x) = 0;

    // 内部遅延器のクリア
    void Clear()
    { for (int n=0; n<delay; n++) un_[n] = 0; }
};    
   
// 櫛形フィルタによる残響生成ユニット
template<int delay> class CombFilter : public Reverb<delay>
{
private:
    const float GC_;
public:
    // コンストラクタ
    CombFilter(float g) : GC_(g) {}

    // 櫛形フィルタの実行
    virtual float Execute(float x)
    {
        float yn = Reverb<delay>::Get();
        Reverb<delay>::Set(x + GC_*yn);
        return yn;
     }
};

// 全域通過フィルタによる残響生成ユニット
template<int delay> class AllpassFilter : public Reverb<delay>
{
private:
    const float G0_;
public:
    // コンストラクタ
    AllpassFilter(float g) : G0_(g) {}

    // 全域通過フィルタの実行
    virtual float Execute(float x)
    {
        float un = x + G0_*Reverb<delay>::Get();
        float yn = -G0_*un + Reverb<delay>::Get();
        Reverb<delay>::Set(un);
        return yn;
     }
};

#endif  // REVERB_UNIT_HPP
