//--------------------------------------------------------------
// Weaver 変調器を使う周波数変換器
//      周波数を 100 Hz 高くする
//      周波数変換器の有効/無効の切り替え：PC の "F446_AD_DA_OnOff.exe"
//
// 2019/01/31, Copyright (c) 2019 MIKAMI, Naoki
//--------------------------------------------------------------

#include "F446_Multirate.hpp"
#include "IIR_Cascade.hpp"
#include "Coefficients.hpp" // フィルタの係数
#include "SerialRxTxIntr.hpp"
#pragma diag_suppress 870   // マルチバイト文字使用の警告抑制のため

using namespace Mikami;

int main()
{
    SerialRxTxIntr rxTx;    // Serial クラスの受送信割込み用オブジェクト
    F446_Multirate myAdDa;  // 出力標本化周波数を４倍にするオブジェクト

    // 入力帯域制限のための帯域通過フィルタ
    IirCascade bpf(ORDER_BP_, cBp_, g0Bpf_);
    // Weaver 変調器で使う低域通過フィルタ
    IirCascade lpfC(ORDER_LP_, cLp_, g0Lpf_);
    IirCascade lpfS(ORDER_LP_, cLp_, g0Lpf_);
    
    const int FS = 10000;                       // 入力の標本化周波数： 10 kHz
    const float PI2 = 3.1415926536f*2;
    const float FC = 2300.0f;                   // 中心周波数
    const float DPHI1 = FC*PI2/FS;              // 2300 Hz
    const float DPHI2 = (100.0f + FC)*PI2/FS;   // 2300 + 100 Hz

    float phi1 = 0;
    float phi2 = 0;
    bool sw = true;     // 周波数変換器の有効/無効のスイッチ

    // F446_Multirate::Start() と F446_Multirate::Input() の間に
    // printf() のように重い処理は実行しないこと．
    myAdDa.Start(FS);       // 標本化を開始する

    while (true)
    {
        //------------------------------------------------------------
        // ここにディジタルフィルタ等の処理を記述する
        float xn = myAdDa.Input();  // 入力

        xn = bpf.Execute(xn);       // 入力帯域制限

        float mpyC = xn*cosf(phi1); // ωc+ωA, ωc-ωA
        float mpyS = xn*sinf(phi1);
        
        // LPF
        mpyC = lpfC.Execute(mpyC);  // ωc-ωA
        mpyS = lpfS.Execute(mpyS);
        
        mpyC = mpyC*cosf(phi2);     // ωsh+2ωc-ωA
        mpyS = mpyS*sinf(phi2);
        
        float yn = 2.0f*(mpyC + mpyS);

        if (sw) myAdDa.Output(yn);  // 出力：周波数変換された信号
        else    myAdDa.Output(xn);  // 出力：入力そのまま
        phi1 = phi1 + DPHI1;
        if (phi1 > PI2) phi1 = phi1 - PI2;
        phi2 = phi2 + DPHI2;
        if (phi2 > PI2) phi2 = phi2 - PI2;
        //------------------------------------------------------------

        //------------------------------------------------------------
        // PC からの指令に対応する処理
        if (rxTx.IsEol())           // 受信バッファのデータが有効になった場合の処理
        {
            string str = rxTx.GetBuffer();
            if (str == "ENQ")
                rxTx.Tx("ACK\n");   // PC からの "ENQ" に対して "ACK" を送信する
            else    // "ENQ" 以外の処理
            {
                if (str == "ST")    // PC から sw の状態の問い合わせ
                {
                    if (sw) rxTx.Tx("EN");     // 周波数変換器が有効であることを返信
                    else    rxTx.Tx("DS");     // 周波数変換器が無効であることを返信
                }
                if (str == "ON")  sw = true;
                if (str == "OFF") sw = false;  
            }
        }
        // PC からの指令に対応する処理はここまで
        //------------------------------------------------------------

    }
}
